<?php
/**
 * @version	$Id: subscription_manager.php 16513 2017-01-20 14:10:53Z alex $
 * @package	In-Portal
 * @copyright	Copyright (C) 1997 - 2012 Intechnic. All rights reserved.
 * @license	  GNU/GPL
 * In-Portal is Open Source software.
 * This means that this software may have been modified pursuant
 * the GNU General Public License, and as distributed it includes
 * or is derivative of works licensed under the GNU General Public License
 * or other free or open source software licenses.
 * See http://www.in-portal.org/license for copyright notices and details.
 */

defined('FULL_PATH') or die('restricted access!');

class kSubscriptionManager extends kBase {

	/**
	 * List of subscriptions (instances of kSubcriptionItem objects)
	 *
	 * @var Array
	 * @access protected
	 */
	protected $subscriptions = Array ();

	/**
	 * Adds set of fields, that uniquely identifies subscription
	 *
	 * @param Array $fields
	 * @return void
	 */
	public function add($fields)
	{
		$this->subscriptions[] = new kSubscriptionItem($fields);
	}

	/**
	 * Detects if current user is subscribed to new posts in given topic
	 *
	 * @return bool
	 * @access public
	 */
	public function subscribed()
	{
		foreach ($this->subscriptions as $subscription) {
			if ( !$subscription->getSubscription()->isLoaded() ) {
				/** @var kSubscriptionItem $subscription */

				return false;
			}
		}

		return true;
	}

	/**
	 * Subscribes current user to new posts in a given topic
	 *
	 * @return bool
	 * @access public
	 */
	public function subscribe()
	{
		foreach ($this->subscriptions as $subscription) {
			if ( !$subscription->subscribe() ) {
				/** @var kSubscriptionItem $subscription */

				return false;
			}
		}

		return true;
	}

	/**
	 * Unsubscribes current user from reciving e-mails about new posts in a gvein topic
	 *
	 * @return bool
	 * @access public
	 */
	public function unsubscribe()
	{
		foreach ($this->subscriptions as $subscription) {
			if ( !$subscription->unsubscribe() ) {
				/** @var kSubscriptionItem $subscription */

				return false;
			}
		}

		return true;
	}

	/**
	 * Returns e-mail event id or throws an exception, when such event not found
	 *
	 * @param string $template_name
	 * @param int $type
	 * @return string
	 * @throws Exception
	 * @access public
	 */
	public function getEmailTemplateId($template_name, $type = EmailTemplate::TEMPLATE_TYPE_FRONTEND)
	{
		$sql = 'SELECT TemplateId
				FROM ' . $this->Application->getUnitOption('email-template', 'TableName') . '
				WHERE TemplateName = ' . $this->Conn->qstr($template_name) . ' AND Type = ' . $type;
		$id = $this->Conn->GetOne($sql);

		if ( !$id ) {
			throw new Exception('E-mail template "' . $template_name . '" not found');
		}

		return $id;
	}
}


class kSubscriptionItem extends kBase {

	/**
	 * Fields set, that uniquely identifies subscription
	 *
	 * @var Array
	 * @access protected
	 */
	protected $fields = Array ();

	/**
	 * Creates new subscription item
	 *
	 * @param Array $fields
	 * @access public
	 */
	public function __construct($fields)
	{
		parent::__construct();

		$this->fields = $fields;
	}

	/**
	 * Returns user subscription
	 *
	 * @param bool $reload
	 * @return kBase|kDBItem
	 * @access public
	 */
	public function getSubscription($reload = false)
	{
		$special = kUtil::crc32(serialize($this->fields));

		/** @var kDBItem $subscription */
		$subscription = $this->Application->recallObject('system-event-subscription.' . $special, null, Array ('skip_autoload' => true));

		if ( !$subscription->isLoaded() || $reload ) {
			$subscription->Load($this->fields);
		}

		return $subscription;
	}

	/**
	 * Subscribes user
	 *
	 * @return bool
	 * @access public
	 */
	public function subscribe()
	{
		$subscription = $this->getSubscription();

		if ( $subscription->isLoaded() ) {
			return true;
		}

		$subscription->SetDBFieldsFromHash($this->fields);

		return $subscription->Create();
	}

	/**
	 * Unsubscribes user
	 *
	 * @return bool
	 * @access public
	 */
	public function unsubscribe()
	{
		$subscription = $this->getSubscription();

		if ( !$subscription->isLoaded() ) {
			return true;
		}

		/** @var kTempTablesHandler $temp_handler */
		$temp_handler = $this->Application->recallObject($subscription->getPrefixSpecial() . '_TempHandler', 'kTempTablesHandler');

		$temp_handler->DeleteItems($subscription->Prefix, $subscription->Special, Array ($subscription->GetID()));

		return true;
	}
}