<?php
/**
* @version	$Id: orders_event_handler.php 16469 2016-11-24 20:49:47Z alex $
* @package	In-Commerce
* @copyright	Copyright (C) 1997 - 2009 Intechnic. All rights reserved.
* @license	Commercial License
* This software is protected by copyright law and international treaties.
* Unauthorized reproduction or unlicensed usage of the code of this program,
* or any portion of it may result in severe civil and criminal penalties,
* and will be prosecuted to the maximum extent possible under the law
* See http://www.in-portal.org/commercial-license for copyright notices and details.
*/

defined('FULL_PATH') or die('restricted access!');

class OrdersEventHandler extends kDBEventHandler
{

	/**
	 * Checks user permission to execute given $event
	 *
	 * @param kEvent $event
	 * @return bool
	 * @access public
	 */
	public function CheckPermission(kEvent $event)
	{
		if ( !$this->Application->isAdminUser ) {
			if ( $event->Name == 'OnCreate' ) {
				// user can't initiate custom order creation directly
				return false;
			}

			$user_id = $this->Application->RecallVar('user_id');
			$items_info = $this->Application->GetVar($event->getPrefixSpecial(true));
			if ( $items_info ) {
				// when POST is present, then check when is beeing submitted
				$order_session_id = $this->Application->RecallVar($event->getPrefixSpecial(true) . '_id');

				$order_dummy = $this->Application->recallObject($event->Prefix . '.-item', null, Array ('skip_autoload' => true));
				/* @var $order_dummy OrdersItem */

				foreach ($items_info as $id => $field_values) {
					if ( $order_session_id != $id ) {
						// user is trying update not his order, even order from other guest
						return false;
					}

					$order_dummy->Load($id);

					// session_id matches order_id from submit
					if ( $order_dummy->GetDBField('PortalUserId') != $user_id ) {
						// user performs event on other user order
						return false;
					}

					$status_field = $order_dummy->getStatusField();

					if ( isset($field_values[$status_field]) && $order_dummy->GetDBField($status_field) != $field_values[$status_field] ) {
						// user can't change status by himself
						return false;
					}

					if ( $order_dummy->GetDBField($status_field) != ORDER_STATUS_INCOMPLETE ) {
						// user can't edit orders being processed
						return false;
					}
				}
			}
		}

		if ( $event->Name == 'OnQuietPreSave' ) {
			$section = $event->getSection();

			if ( $this->isNewItemCreate($event) ) {
				return $this->Application->CheckPermission($section . '.add', 1);
			}
			else {
				return $this->Application->CheckPermission($section . '.add', 1) || $this->Application->CheckPermission($section . '.edit', 1);
			}
		}

		return parent::CheckPermission($event);
	}

	/**
	 * Allows to override standard permission mapping
	 *
	 * @return void
	 * @access protected
	 * @see kEventHandler::$permMapping
	 */
	protected function mapPermissions()
	{
		parent::mapPermissions();

		$permissions = Array (
			// admin
			'OnRecalculateItems'	=>	Array('self' => 'add|edit'),
			'OnResetToUser'			=>	Array('self' => 'add|edit'),
			'OnResetToBilling'		=>	Array('self' => 'add|edit'),
			'OnResetToShipping'		=>	Array('self' => 'add|edit'),
			'OnMassOrderApprove'	=>	Array('self' => 'advanced:approve'),
			'OnMassOrderDeny'		=>	Array('self' => 'advanced:deny'),
			'OnMassOrderArchive'	=>	Array('self' => 'advanced:archive'),
			'OnMassPlaceOrder'		=>	Array('self' => 'advanced:place'),
			'OnMassOrderProcess'	=>	Array('self' => 'advanced:process'),
			'OnMassOrderShip'		=>	Array('self' => 'advanced:ship'),
			'OnResetToPending'		=>	Array('self' => 'advanced:reset_to_pending'),
			'OnLoadSelected'		=>	Array('self' => 'view'),	// print in this case
			'OnGoToOrder'			=>	Array('self' => 'view'),

			// front-end
			'OnViewCart'			=>	Array('self' => true),
			'OnAddToCart'			=>	Array('self' => true),
			'OnRemoveFromCart'		=>	Array('self' => true),
			'OnUpdateCart'			=>	Array('self' => true),
			'OnUpdateCartJSON'		=>	Array('self' => true),
			'OnUpdateItemOptions'	=>	Array('self' => true),
			'OnCleanupCart'			=>	Array('self' => true),
			'OnContinueShopping'	=>	Array('self' => true),
			'OnCheckout'			=>	Array('self' => true),
			'OnSelectAddress'		=>	Array('self' => true),
			'OnProceedToBilling'	=>	Array('self' => true),
			'OnProceedToPreview'	=>	Array('self' => true),
			'OnProceedToPreviewAjax' => array('self' => true),
			'OnCompleteOrder'		=>	Array('self' => true),
			'OnUpdate' => array('self' => true),
			'OnUpdateAjax'			=>	Array('self' => true),

			'OnRemoveCoupon'		=>	Array('self' => true),
			'OnRemoveGiftCertificate'		=>	Array('self' => true),

			'OnCancelRecurring'		=>	Array('self' => true),
			'OnAddVirtualProductToCart'		=>	Array('self' => true),
			'OnItemBuild'		=>	Array('self' => true),
			'OnDownloadLabel' 	=>  Array('self' => true, 'subitem' => true),
		);

		$this->permMapping = array_merge($this->permMapping, $permissions);
	}

	/**
	 * Define alternative event processing method names
	 *
	 * @return void
	 * @see kEventHandler::$eventMethods
	 * @access protected
	 */
	protected function mapEvents()
	{
		parent::mapEvents();

		$common_events = Array (
			'OnResetToUser'		=>	'OnResetAddress',
			'OnResetToBilling'	=>	'OnResetAddress',
			'OnResetToShipping'	=>	'OnResetAddress',

			'OnMassOrderProcess'	=>	'MassInventoryAction',
			'OnMassOrderApprove'	=>	'MassInventoryAction',
			'OnMassOrderDeny'		=>	'MassInventoryAction',
			'OnMassOrderArchive'	=>	'MassInventoryAction',
			'OnMassOrderShip'		=>	'MassInventoryAction',

			'OnOrderProcess'	=>	'InventoryAction',
			'OnOrderApprove'	=>	'InventoryAction',
			'OnOrderDeny'		=>	'InventoryAction',
			'OnOrderArchive'	=>	'InventoryAction',
			'OnOrderShip'		=>	'InventoryAction',
		);

		$this->eventMethods = array_merge($this->eventMethods, $common_events);
	}

	/* ======================== FRONT ONLY ======================== */

	function OnQuietPreSave($event)
	{
		$object = $event->getObject();
		/* @var $object kDBItem */

		$object->IgnoreValidation = true;
		$event->CallSubEvent('OnPreSave');
		$object->IgnoreValidation = false;
	}

	/**
	 * Sets new address to order
	 *
	 * @param kEvent $event
	 */
	function OnSelectAddress($event)
	{
		if ($this->Application->isAdminUser) {
			return ;
		}

		$object = $event->getObject();
		/* @var $object OrdersItem */

		$shipping_address_id = $this->Application->GetVar('shipping_address_id');
		$billing_address_id = $this->Application->GetVar('billing_address_id');

		if ($shipping_address_id || $billing_address_id) {
			$cs_helper = $this->Application->recallObject('CountryStatesHelper');
			/* @var $cs_helper kCountryStatesHelper */

			$address = $this->Application->recallObject('addr.-item','addr', Array('skip_autoload' => true));
			/* @var $address AddressesItem */

			$addr_list = $this->Application->recallObject('addr', 'addr_List', Array('per_page'=>-1, 'skip_counting'=>true) );
			/* @var $addr_list AddressesList */

			$addr_list->Query();
		}

		if ($shipping_address_id > 0) {
			$addr_list->CopyAddress($shipping_address_id, 'Shipping');
			$address->Load($shipping_address_id);
			$address->MarkAddress('Shipping');

			$cs_helper->PopulateStates($event, 'ShippingState', 'ShippingCountry');
			$object->setRequired('ShippingState', false);
		}
		elseif ($shipping_address_id == -1) {
			$object->ResetAddress('Shipping');
		}

		if ($billing_address_id > 0) {
			$addr_list->CopyAddress($billing_address_id, 'Billing');
			$address->Load($billing_address_id);
			$address->MarkAddress('Billing');

			$cs_helper->PopulateStates($event, 'BillingState', 'BillingCountry');
			$object->setRequired('BillingState', false);
		}
		elseif ($billing_address_id == -1) {
			$object->ResetAddress('Billing');
		}

		$event->redirect = false;

		$object->IgnoreValidation = true;
		$this->RecalculateTax($event);
		$object->Update();
	}

	/**
	 * Updates order with registred user id
	 *
	 * @param kEvent $event
	 */
	function OnUserCreate($event)
	{
		if( !($event->MasterEvent->status == kEvent::erSUCCESS) ) return false;

		$ses_id = $this->Application->RecallVar('front_order_id');
		if($ses_id)
		{
			$this->updateUserID($ses_id, $event);
			$this->Application->RemoveVar('front_order_id');
		}
	}

	/**
	 * Updates shopping cart with logged-in user details
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function OnUserLogin($event)
	{
		if ( ($event->MasterEvent->status != kEvent::erSUCCESS) || kUtil::constOn('IS_INSTALL') ) {
			// login failed OR login during installation
			return;
		}

		$ses_id = $this->Application->RecallVar('ord_id');

		if ( $ses_id ) {
			$this->updateUserID($ses_id, $event);
		}

		$user_id = $this->Application->RecallVar('user_id');
		$affiliate_id = $this->isAffiliate($user_id);

		if ( $affiliate_id ) {
			$this->Application->setVisitField('AffiliateId', $affiliate_id);
		}

		$event->CallSubEvent('OnRecalculateItems');
	}

	/**
	 * Puts ID of just logged-in user into current order
	 *
	 * @param int $order_id
	 * @param kEvent $event
	 * @return void
	 */
	function updateUserID($order_id, $event)
	{
		$user = $this->Application->recallObject('u.current');
		/* @var $user UsersItem */

		$affiliate_id = $this->isAffiliate( $user->GetID() );

		$fields_hash = Array (
			'PortalUserId' => $user->GetID(),
			'BillingEmail' => $user->GetDBField('Email'),
		);

		if ( $affiliate_id ) {
			$fields_hash['AffiliateId'] = $affiliate_id;
		}

		/** @var OrdersItem $object */
		$object = $this->Application->recallObject($event->Prefix . '.-item', null, array('skip_autoload' => true));
		$object->Load($order_id);

		if ( $object->isLoaded() ) {
			$object->SetDBFieldsFromHash($fields_hash);
			$object->Update();
		}
	}

	function isAffiliate($user_id)
	{
		$affiliate_user = $this->Application->recallObject('affil.-item', null, Array('skip_autoload' => true) );
		/* @var $affiliate_user kDBItem */

		$affiliate_user->Load($user_id, 'PortalUserId');

		return $affiliate_user->isLoaded() ? $affiliate_user->GetDBField('AffiliateId') : 0;
	}

	/**
	 * Charge order
	 *
	 * @param OrdersItem $order
	 * @return Array
	 */
	function ChargeOrder(&$order)
	{
		$gw_data = $order->getGatewayData();

		/** @var kGWBase $gateway_object */
		$gateway_object = $this->Application->recallObject($gw_data['ClassName']);

		$payment_result = $gateway_object->DirectPayment($order->GetFieldValues(), $gw_data['gw_params']);
		$sql = 'UPDATE %s SET GWResult1 = %s WHERE %s = %s';
		$sql = sprintf($sql, $order->TableName, $this->Conn->qstr($gateway_object->getGWResponce()), $order->IDField, $order->GetID() );
		$this->Conn->Query($sql);
		$order->SetDBField('GWResult1', $gateway_object->getGWResponce() );

		return array('result'=>$payment_result, 'data'=>$gateway_object->parsed_responce, 'gw_data' => $gw_data, 'error_msg'=>$gateway_object->getErrorMsg());
	}

	/**
	 * Returns parameters, used to send order-related e-mails
	 *
	 * @param OrdersItem $order
	 * @return array
	 */
	function OrderEmailParams(&$order)
	{
		$billing_email = $order->GetDBField('BillingEmail');

		$sql = 'SELECT Email
				FROM ' . $this->Application->getUnitOption('u', 'TableName') . '
				WHERE PortalUserId = ' . $order->GetDBField('PortalUserId');
		$user_email = $this->Conn->GetOne($sql);

		$ret = Array (
			'_user_email' => $user_email, // for use when shipping vs user is required in InventoryAction
			'to_name' => $order->GetDBField('BillingTo'),
			'to_email' => $billing_email ? $billing_email : $user_email,
		);

		return $ret;
	}

	function PrepareCoupons($event, &$order)
	{
		$order_items = $this->Application->recallObject('orditems.-inv','orditems_List',Array('skip_counting'=>true,'per_page'=>-1) );
		/* @var $order_items kDBList */

		$order_items->linkToParent($order->Special);
		$order_items->Query();
		$order_items->GoFirst();

		$assigned_coupons = array();
		$coup_handler = $this->Application->recallObject('coup_EventHandler');
		foreach($order_items->Records as $product_item)
		{
			if ($product_item['ItemData']) {
				$item_data = unserialize($product_item['ItemData']);
				if (isset($item_data['AssignedCoupon']) && $item_data['AssignedCoupon']) {
					$coupon_id = $item_data['AssignedCoupon'];
					// clone coupon, get new coupon ID
					$coupon = $this->Application->recallObject('coup',null,array('skip_autload' => true));
					/* @var $coupon kDBItem */
					$coupon->Load($coupon_id);
					if (!$coupon->isLoaded()) continue;

					$coup_handler->SetNewCode($coupon);
					$coupon->NameCopy();
					$coupon->SetDBField('Name', $coupon->GetDBField('Name').' (Order #'.$order->GetField('OrderNumber').')');
					$coupon->Create();

					// add coupon code to array
					array_push($assigned_coupons, $coupon->GetDBField('Code'));
				}
			}
		}

		/* @var $order OrdersItem */
		if ($assigned_coupons) {
			$comments = $order->GetDBField('AdminComment');
			if ($comments) $comments .= "\r\n";
			$comments .= "Issued coupon(s): ". join(',', $assigned_coupons);
			$order->SetDBField('AdminComment', $comments);
			$order->Update();
		}

		if ($assigned_coupons) $this->Application->SetVar('order_coupons', join(',', $assigned_coupons));
	}

	/**
	 * Completes order if possible
	 *
	 * @param kEvent $event
	 * @return bool
	 */
	function OnCompleteOrder($event)
	{
		$this->LockTables($event);
		$reoccurring_order = substr($event->Special, 0, 9) == 'recurring';

		if ( !$reoccurring_order && !$this->CheckQuantites($event) ) {
			// don't check quantities (that causes recalculate) for reoccurring orders
			return;
		}

		$this->ReserveItems($event);

		$order = $event->getObject();
		/* @var $order OrdersItem */

		$charge_result = $this->ChargeOrder($order);

		if (!$charge_result['result']) {
			$this->FreeItems($event);
			$this->Application->StoreVar('gw_error', $charge_result['error_msg']);

			//$this->Application->StoreVar('gw_error', getArrayValue($charge_result, 'data', 'responce_reason_text') );
			$event->redirect = $this->Application->GetVar('failure_template');
			$event->SetRedirectParam('m_cat_id', 0);
			if ($event->Special == 'recurring') { // if we set failed status for other than recurring special the redirect will not occur
				$event->status = kEvent::erFAIL;
			}
			return false;
		}

		// call CompleteOrder events for items in order BEFORE SplitOrder (because ApproveEvents are called there)
		$order_items = $this->Application->recallObject('orditems.-inv','orditems_List',Array('skip_counting'=>true,'per_page'=>-1) );
		/* @var $order_items kDBList */

		$order_items->linkToParent($order->Special);
		$order_items->Query(true);
		$order_items->GoFirst();

		foreach($order_items->Records as $product_item)
		{
			if (!$product_item['ProductId']) continue; // product may have been deleted
			$this->raiseProductEvent('CompleteOrder', $product_item['ProductId'], $product_item);
		}

		$shipping_control = getArrayValue($charge_result, 'gw_data', 'gw_params', 'shipping_control');
		if ($event->Special != 'recurring') {
			if ($shipping_control && $shipping_control != SHIPPING_CONTROL_PREAUTH ) {
				// we have to do it here, because the coupons are used in the e-mails
				$this->PrepareCoupons($event, $order);
			}

			$this->Application->emailUser('ORDER.SUBMIT', null, $this->OrderEmailParams($order));
			$this->Application->emailAdmin('ORDER.SUBMIT');
		}

		if ($shipping_control === false || $shipping_control == SHIPPING_CONTROL_PREAUTH ) {
			$order->SetDBField('Status', ORDER_STATUS_PENDING);
			$order->Update();
		}
		else {
			$this->SplitOrder($event, $order);
		}

		if (!$this->Application->isAdminUser) {
			// for tracking code
			$this->Application->StoreVar('last_order_amount', $order->GetDBField('TotalAmount'));
			$this->Application->StoreVar('last_order_number', $order->GetDBField('OrderNumber'));
			$this->Application->StoreVar('last_order_customer', $order->GetDBField('BillingTo'));
			$this->Application->StoreVar('last_order_user', $order->GetDBField('Username'));

			$event->redirect = $this->Application->GetVar('success_template');
			$event->SetRedirectParam('m_cat_id', 0);
		}
		else
		{
//			$event->CallSubEvent('OnSave');
		}

		$order_id = $order->GetId();
		$order_idfield = $this->Application->getUnitOption('ord','IDField');
		$order_table = $this->Application->getUnitOption('ord','TableName');
		$original_amount = $order->GetDBField('SubTotal') + $order->GetDBField('ShippingCost') + $order->GetDBField('VAT') + $order->GetDBField('ProcessingFee') + $order->GetDBField('InsuranceFee') - $order->GetDBField('GiftCertificateDiscount');
		$sql = 'UPDATE '.$order_table.'
				SET OriginalAmount = '.$original_amount.'
				WHERE '.$order_idfield.' = '.$order_id;
		$this->Conn->Query($sql);

		// Remember order ID for use on "Thank You" page.
		$this->Application->StoreVar('front_order_id', $order_id);

		// Remove globals, set from "_createNewCart" method.
		$this->Application->DeleteVar('ord_id');
		$this->Application->RemoveVar('ord_id');

		// Prevent accidental access to non-Incomplete order.
		$this->Application->removeObject($event->getPrefixSpecial());
		$this->Application->Session->SetCookie('shop_cart_cookie', '', strtotime('-1 month'));
	}

	/**
	 * Set billing address same as shipping
	 *
	 * @param kEvent $event
	 */
	function setBillingAddress($event)
	{
		$object = $event->getObject();
		/* @var $object OrdersItem */

		if ( $object->HasTangibleItems() ) {
			if ( $this->Application->GetVar('same_address') ) {
				// copy shipping address to billing
				$items_info = $this->Application->GetVar($event->getPrefixSpecial(true));
				list($id, $field_values) = each($items_info);

				$address_fields = Array (
					'To', 'Company', 'Phone', 'Fax', 'Email',
					'Address1', 'Address2', 'City', 'State',
					'Zip', 'Country'
				);

				foreach ($address_fields as $address_field) {
					$items_info[$id]['Billing' . $address_field] = $object->GetDBField('Shipping' . $address_field);
				}

				$this->Application->SetVar($event->getPrefixSpecial(true), $items_info);
			}
		}
	}

	/**
	 * Enter description here...
	 *
	 * @param kEvent $event
	 */
	function OnProceedToPreview($event)
	{
		$this->setBillingAddress($event);

		$event->CallSubEvent('OnUpdate');
		$event->redirect = $this->Application->GetVar('preview_template');
	}


	function OnViewCart($event)
	{
		$this->StoreContinueShoppingLink();
		$event->redirect = $this->Application->GetVar('viewcart_template');
	}

	function OnContinueShopping($event)
	{
		$order_helper = $this->Application->recallObject('OrderHelper');
		/* @var $order_helper OrderHelper */

		$template = $this->Application->GetVar('continue_shopping_template');

		$event->redirect = $order_helper->getContinueShoppingTemplate($template);
	}

	/**
	 * Enter description here...
	 *
	 * @param kEvent $event
	 */
	function OnCheckout($event)
	{
		$this->OnUpdateCart($event);
		if ( !$event->getEventParam('RecalculateChangedCart') ) {
			$object = $event->getObject();
			/* @var $object OrdersItem */

			if ( !$object->HasTangibleItems() ) {
				$object->SetDBField('ShippingTo', '');
				$object->SetDBField('ShippingCompany', '');
				$object->SetDBField('ShippingPhone', '');
				$object->SetDBField('ShippingFax', '');
				$object->SetDBField('ShippingEmail', '');
				$object->SetDBField('ShippingAddress1', '');
				$object->SetDBField('ShippingAddress2', '');
				$object->SetDBField('ShippingCity', '');
				$object->SetDBField('ShippingState', '');
				$object->SetDBField('ShippingZip', '');
				$object->SetDBField('ShippingCountry', '');
				$object->SetDBField('ShippingType', 0);
				$object->SetDBField('ShippingCost', 0);
				$object->SetDBField('ShippingCustomerAccount', '');
				$object->SetDBField('ShippingTracking', '');
				$object->SetDBField('ShippingDate', 0);
				$object->SetDBField('ShippingOption', 0);
				$object->SetDBField('ShippingInfo', '');
				$object->Update();
			}

			$event->redirect = $this->Application->GetVar('next_step_template');

			$order_id = $this->Application->GetVar('order_id');

			if ( $order_id !== false ) {
				$event->SetRedirectParam('ord_id', $order_id);
			}
		}
	}

	/**
	 * Restores order from cookie
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function OnRestoreOrder(kEvent $event)
	{
		if ( $this->Application->isAdmin || $this->Application->RecallVar('ord_id') ) {
			// admin OR there is an active order -> don't restore from cookie
			return;
		}

		$shop_cart_cookie = $this->Application->GetVarDirect('shop_cart_cookie', 'Cookie');

		if ( !$shop_cart_cookie ) {
			return;
		}

		$user_id = $this->Application->RecallVar('user_id');

		$sql = 'SELECT OrderId
				FROM ' . TABLE_PREFIX . 'Orders
				WHERE (OrderId = ' . (int)$shop_cart_cookie . ') AND (Status = ' . ORDER_STATUS_INCOMPLETE . ') AND (PortalUserId = ' . $user_id . ')';
		$order_id = $this->Conn->GetOne($sql);

		if ( $order_id ) {
			$this->Application->StoreVar('ord_id', $order_id);
		}
	}

	/**
	 * Redirect user to Billing checkout step
	 *
	 * @param kEvent $event
	 */
	function OnProceedToBilling($event)
	{
		$items_info = $this->Application->GetVar($event->getPrefixSpecial(true));
		if ( $items_info ) {
			list($id, $field_values) = each($items_info);

			$object = $event->getObject();
			/* @var $object kDBItem */

			$payment_type_id = $object->GetDBField('PaymentType');

			if ( !$payment_type_id ) {
				$default_type = $this->_getDefaultPaymentType();

				if ( $default_type ) {
					$field_values['PaymentType'] = $default_type;
					$items_info[$id] = $field_values;
					$this->Application->SetVar($event->getPrefixSpecial(true), $items_info);
				}
			}
		}

		$event->CallSubEvent('OnUpdate');
		$event->redirect = $this->Application->GetVar('next_step_template');
	}

	/**
	 * Removes reoccurring mark from the order
	 *
	 * @param kEvent $event
	 * @return void
	 */
	protected function OnCancelRecurring($event)
	{
		$order = $event->getObject();
		/* @var $order OrdersItem */

		$order->SetDBField('IsRecurringBilling', 0);
		$order->Update();

		if ( $this->Application->GetVar('cancelrecurring_ok_template') ) {
			$event->redirect = $this->Application->GetVar('cancelrecurring_ok_template');
		}
	}

	/**
	 * Occurs after updating item
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function OnAfterItemUpdate(kEvent $event)
	{
		parent::OnAfterItemUpdate($event);

		$object = $event->getObject();
		/* @var $object OrdersItem */

		$cvv2 = $object->GetDBField('PaymentCVV2');

		if ( $cvv2 !== false ) {
			$this->Application->StoreVar('CVV2Code', $cvv2);
		}
	}


	/**
	 * Updates kDBItem
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function OnUpdate(kEvent $event)
	{
		$this->setBillingAddress($event);

		parent::OnUpdate($event);

		if ($this->Application->isAdminUser) {
			return ;
		}
		else {
			$event->SetRedirectParam('opener', 's');
		}

		if ($event->status == kEvent::erSUCCESS) {
			$this->createMissingAddresses($event);
		}
		else {
			// strange: recalculate total amount on error
			$object = $event->getObject();
			/* @var $object OrdersItem */

			$object->SetDBField('TotalAmount', $object->getTotalAmount());
		}
	}

	/**
	 * Creates new address
	 *
	 * @param kEvent $event
	 */
	function createMissingAddresses($event)
	{
		if ( !$this->Application->LoggedIn() ) {
			return ;
		}

		$object = $event->getObject();
		/* @var $object kDBItem */

		$addr_list = $this->Application->recallObject('addr', 'addr_List', Array ('per_page' => -1, 'skip_counting' => true));
		/* @var $addr_list kDBList */

		$addr_list->Query();

		$address_dummy = $this->Application->recallObject('addr.-item', null, Array ('skip_autoload' => true));
		/* @var $address_dummy AddressesItem */

		$address_prefixes = Array ('Billing', 'Shipping');
		$address_fields = Array (
			'To', 'Company', 'Phone', 'Fax', 'Email', 'Address1',
			'Address2', 'City', 'State', 'Zip', 'Country'
		);

		foreach ($address_prefixes as $address_prefix) {
			$address_id = $this->Application->GetVar(strtolower($address_prefix) . '_address_id');

			if ( !$this->Application->GetVar('check_' . strtolower($address_prefix) . '_address') ) {
				// form type doesn't match check type, e.g. shipping check on billing form
				continue;
			}

			if ( $address_id > 0 ) {
				$address_dummy->Load($address_id);
			}
			else {
				$address_dummy->SetDBField('PortalUserId', $this->Application->RecallVar('user_id'));
			}

			foreach ($address_fields as $address_field) {
				$address_dummy->SetDBField($address_field, $object->GetDBField($address_prefix . $address_field));
			}

			$address_dummy->MarkAddress($address_prefix, false);

			$ret = ($address_id > 0) ? $address_dummy->Update() : $address_dummy->Create();
		}
	}

	/**
	 * Updates shopping cart content
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function OnUpdateCart($event)
	{
		$this->Application->HandleEvent(new kEvent('orditems:OnUpdate'));

		$event->CallSubEvent('OnRecalculateItems');
	}

	/**
	 * Updates cart and returns various info in JSON format
	 *
	 * @param kEvent $event
	 */
	function OnUpdateCartJSON($event)
	{
		if ( $this->Application->GetVar('ajax') != 'yes' ) {
			return;
		}

		$object = $event->getObject();
		/* @var $object kDBItem */

		// 1. delete given order item by id
		$delete_id = $this->Application->GetVar('delete_id');

		if ( $delete_id !== false ) {
			$sql = 'DELETE FROM ' . TABLE_PREFIX . 'OrderItems
					WHERE OrderId = ' . $object->GetID() . ' AND OrderItemId = ' . (int)$delete_id;
			$this->Conn->Query($sql);
		}

		// 2. remove coupon
		$remove = $this->Application->GetVar('remove');

		if ( $remove == 'coupon' ) {
			$this->RemoveCoupon($object);
			$object->setCheckoutError(OrderCheckoutErrorType::COUPON, OrderCheckoutError::COUPON_REMOVED);
		}
		elseif ( $remove == 'gift_certificate' ) {
			$this->RemoveGiftCertificate($object);
			$object->setCheckoutError(OrderCheckoutErrorType::GIFT_CERTIFICATE, OrderCheckoutError::GC_REMOVED);
		}

		// 3. update product quantities and recalculate all discounts
		$this->Application->HandleEvent(new kEvent('orditems:OnUpdate'));
		$event->CallSubEvent('OnRecalculateItems');

		// 4. remove "orditems" object of kDBItem class, since getOrderInfo uses kDBList object under same prefix
		$this->Application->removeObject('orditems');

		$order_helper = $this->Application->recallObject('OrderHelper');
		/* @var $order_helper OrderHelper */

		$event->status = kEvent::erSTOP;
		$currency = $this->Application->GetVar('currency', 'selected');

		echo json_encode( $order_helper->getOrderInfo($object, $currency) );
	}

	/**
	 * Adds item to cart
	 *
	 * @param kEvent $event
	 */
	function OnAddToCart($event)
	{
		$this->StoreContinueShoppingLink();

		$qty = $this->Application->GetVar('qty');
		$options = $this->Application->GetVar('options');

		// multiple or options add
		$items = Array();
		if (is_array($qty)) {
			foreach ($qty as $item_id => $combinations)
			{
				if (is_array($combinations)) {
					foreach ($combinations as $comb_id => $comb_qty) {
						if ($comb_qty == 0) continue;
						$items[] = array('item_id' => $item_id, 'qty' => $comb_qty, 'comb' => $comb_id);
					}
				}
				else {
					$items[] = array('item_id' => $item_id, 'qty' => $combinations);
				}
			}
		}

		if (!$items) {
			if (!$qty || is_array($qty)) $qty = 1;
			$item_id = $this->Application->GetVar('p_id');
			if (!$item_id) return ;
			$items = array(array('item_id' => $item_id, 'qty' => $qty));
		}

		// remember item data passed to event when called
		$default_item_data = $event->getEventParam('ItemData');
		$default_item_data = $default_item_data ? unserialize($default_item_data) : Array();

		foreach ($items as $an_item) {
			$item_id = $an_item['item_id'];
			$qty = $an_item['qty'];
			$comb = getArrayValue($an_item, 'comb');

			$item_data = $default_item_data;

			$product = $this->Application->recallObject('p', null, Array('skip_autoload' => true));
			/* @var $product ProductsItem */

			$product->Load($item_id);

			$event->setEventParam('ItemData', null);

			if ($product->GetDBField('AssignedCoupon')) {
				$item_data['AssignedCoupon'] = $product->GetDBField('AssignedCoupon');
			}

			// 1. store options information OR
			if ($comb) {
				$combination = $this->Conn->GetOne('SELECT Combination FROM '.TABLE_PREFIX.'ProductOptionCombinations WHERE CombinationId = '.$comb);
				$item_data['Options'] = unserialize($combination);
			}
			elseif (is_array($options)) {
				$item_data['Options'] = $options[$item_id];
			}

			// 2. store subscription information OR
			if( $product->GetDBField('Type') == 2 )	// subscriptions
			{
				$item_data = $this->BuildSubscriptionItemData($item_id, $item_data);
			}

			// 3. store package information
			if( $product->GetDBField('Type') == 5 )	// package
			{
				$package_content_ids = $product->GetPackageContentIds();

				$product_package_item = $this->Application->recallObject('p.-packageitem');
				/* @var $product_package_item ProductsItem */

				$package_item_data = array();

				foreach ($package_content_ids as $package_item_id){
					$product_package_item->Load($package_item_id);
					$package_item_data[$package_item_id] = array();
					if( $product_package_item->GetDBField('Type') == 2 )	// subscriptions
					{
						$package_item_data[$package_item_id] = $this->BuildSubscriptionItemData($package_item_id, $item_data);
					}
				}

				$item_data['PackageContent'] = $product->GetPackageContentIds();
				$item_data['PackageItemsItemData'] = $package_item_data;
			}

			$event->setEventParam('ItemData', serialize($item_data));
			// 1 for PacakgeNum when in admin - temporary solution to overcome splitting into separate sub-orders
			// of orders with items added through admin when approving them
			$this->AddItemToOrder($event, $item_id, $qty, $this->Application->isAdminUser ? 1 : null);
		}
		if ($event->status == kEvent::erSUCCESS && !$event->redirect) {
			$event->SetRedirectParam('pass', 'm');
			$event->SetRedirectParam('pass_category', 0); //otherwise mod-rewrite shop-cart URL will include category
			$event->redirect = true;
		}
		else {
			if ($this->Application->isAdminUser) {
				$event->SetRedirectParam('opener', 'u');
			}
		}
	}

	/**
	 * Returns table prefix from event (temp or live)
	 *
	 * @param kEvent $event
	 * @return string
	 * @todo Needed? Should be refactored (by Alex)
	 */
	function TablePrefix(kEvent $event)
	{
		return $this->UseTempTables($event) ? $this->Application->GetTempTablePrefix('prefix:' . $event->Prefix) . TABLE_PREFIX : TABLE_PREFIX;
	}

	/**
	 * Check if required options are selected & selected option combination is in stock
	 *
	 * @param kEvent $event
	 * @param Array $options
	 * @param int $product_id
	 * @param int $qty
	 * @param int $selection_mode
	 * @return bool
	 */
	function CheckOptions($event, &$options, $product_id, $qty, $selection_mode)
	{
		// 1. check for required options
		$selection_filter = $selection_mode == 1 ? ' AND OptionType IN (1,3,6) ' : '';
		$req_options = $this->Conn->GetCol('SELECT ProductOptionId FROM '.TABLE_PREFIX.'ProductOptions WHERE ProductId = '.$product_id.' AND Required = 1 '.$selection_filter);
		$result = true;
		foreach ($req_options as $opt_id) {
			if (!getArrayValue($options, $opt_id)) {
				$this->Application->SetVar('opt_error', 1); //let the template know we have an error
				$result = false;
			}
		}

		// 2. check for option combinations in stock
		$comb_salt = $this->OptionsSalt($options, true);
		if ($comb_salt) {
			// such option combination is defined explicitly
			$poc_table = $this->Application->getUnitOption('poc', 'TableName');
			$sql = 'SELECT Availability
					FROM '.$poc_table.'
					WHERE CombinationCRC = '.$comb_salt;
			$comb_availble = $this->Conn->GetOne($sql);

			// 2.1. check if Availability flag is set, then
			if ($comb_availble == 1) {
				// 2.2. check for quantity in stock
				$table = Array();
				$table['poc'] = $this->Application->getUnitOption('poc', 'TableName');
				$table['p'] = $this->Application->getUnitOption('p', 'TableName');
				$table['oi'] = $this->TablePrefix($event).'OrderItems';

				$object = $event->getObject();
				$ord_id = $object->GetID();

				// 2.3. check if some amount of same combination & product are not already in shopping cart
				$sql = 'SELECT '.
								$table['p'].'.InventoryStatus,'.
								$table['p'].'.BackOrder,
								IF('.$table['p'].'.InventoryStatus = 2, '.$table['poc'].'.QtyInStock, '.$table['p'].'.QtyInStock) AS QtyInStock,
								IF('.$table['oi'].'.OrderItemId IS NULL, 0, '.$table['oi'].'.Quantity) AS Quantity
						FROM '.$table['p'].'
						LEFT JOIN '.$table['poc'].' ON
								'.$table['p'].'.ProductId = '.$table['poc'].'.ProductId
						LEFT JOIN '.$table['oi'].' ON
								('.$table['oi'].'.OrderId = '.$ord_id.') AND
								('.$table['oi'].'.OptionsSalt = '.$comb_salt.') AND
								('.$table['oi'].'.ProductId = '.$product_id.') AND
								('.$table['oi'].'.BackOrderFlag = 0)
						WHERE '.$table['poc'].'.CombinationCRC = '.$comb_salt;
				$product_info = $this->Conn->GetRow($sql);

				if ($product_info['InventoryStatus']) {
					$backordering = $this->Application->ConfigValue('Comm_Enable_Backordering');
					if (!$backordering || $product_info['BackOrder'] == 0) {
						// backordering is not enabled generally or for this product directly, then check quantities in stock
						if ($qty + $product_info['Quantity'] > $product_info['QtyInStock']) {
							$this->Application->SetVar('opt_error', 2);
							$result = false;
						}
					}
				}
			}
			elseif ($comb_availble !== false) {
				$this->Application->SetVar('opt_error', 2);
				$result = false;
			}
		}

		if ($result) {
			$event->status = kEvent::erSUCCESS;
			$shop_cart_template = $this->Application->GetVar('shop_cart_template');
			$event->redirect = $this->Application->isAdminUser || !$shop_cart_template ? true : $shop_cart_template;
		}
		else {
			$event->status = kEvent::erFAIL;
		}
		return $result;
	}

	/**
	 * Enter description here...
	 *
	 * @param kEvent $event
	 */
	function OnUpdateItemOptions($event)
	{
		$opt_data = $this->Application->GetVar('options');
		$options = getArrayValue($opt_data, $this->Application->GetVar('p_id'));

		if (!$options) {
			$qty_data = $this->Application->GetVar('qty');
			$comb_id = key(getArrayValue($qty_data, $this->Application->GetVar('p_id')));
			$options = unserialize($this->Conn->GetOne('SELECT Combination FROM '.TABLE_PREFIX.'ProductOptionCombinations WHERE CombinationId = '.$comb_id));
		}

		if (!$options) return;

		$ord_item = $this->Application->recallObject('orditems.-opt', null, Array ('skip_autoload' => true));
		/* @var $ord_item kDBItem */

		$ord_item->Load($this->Application->GetVar('orditems_id'));

		// assuming that quantity cannot be changed during order item editing
		if (!$this->CheckOptions($event, $options, $ord_item->GetDBField('ProductId'), 0, $ord_item->GetDBField('OptionsSelectionMode'))) return;

		$item_data = unserialize($ord_item->GetDBField('ItemData'));
		$item_data['Options'] = $options;
		$ord_item->SetDBField('ItemData', serialize($item_data));
		$ord_item->SetDBField('OptionsSalt', $this->OptionsSalt($options));
		$ord_item->Update();
		$event->CallSubEvent('OnRecalculateItems');
		if ($event->status == kEvent::erSUCCESS && $this->Application->isAdminUser) {
			$event->SetRedirectParam('opener', 'u');
		}
	}

	function BuildSubscriptionItemData($item_id, $item_data)
	{
		$products_table = $this->Application->getUnitOption('p', 'TableName');
		$products_idfield = $this->Application->getUnitOption('p', 'IDField');
		$sql = 'SELECT AccessGroupId FROM %s WHERE %s = %s';
		$item_data['PortalGroupId'] = $this->Conn->GetOne( sprintf($sql, $products_table, $products_idfield, $item_id) );

		$pricing_table = $this->Application->getUnitOption('pr', 'TableName');
		$pricing_idfield = $this->Application->getUnitOption('pr', 'IDField');

		/* TODO check on implementation
		$sql = 'SELECT AccessDuration, AccessUnit, DurationType, AccessExpiration FROM %s WHERE %s = %s';
		*/

		$sql = 'SELECT * FROM %s WHERE %s = %s';
		$pricing_id = $this->GetPricingId($item_id, $item_data);
		$item_data['PricingId'] = $pricing_id;

		$pricing_info = $this->Conn->GetRow( sprintf($sql, $pricing_table, $pricing_idfield, $pricing_id ) );
		$unit_secs = Array(1 => 1, 2 => 60, 3 => 3600, 4 => 86400, 5 => 604800, 6 => 2592000, 7 => 31536000);

		/* TODO check on implementation (code from customization healtheconomics.org)
		$item_data['DurationType'] = $pricing_info['DurationType'];
		$item_data['AccessExpiration'] = $pricing_info['AccessExpiration'];
		*/

		$item_data['Duration'] = $pricing_info['AccessDuration'] * $unit_secs[ $pricing_info['AccessUnit'] ];

		return $item_data;
	}

	/**
	 * Enter description here...
	 *
	 * @param kEvent $event
	 */
	function OnApplyCoupon($event)
	{
		$code = $this->Application->GetVar('coupon_code');

		if ($code == '') {
			return ;
		}

		$object = $event->getObject();
		/* @var $object OrdersItem */

		$coupon = $this->Application->recallObject('coup', null, Array ('skip_autoload' => true));
		/* @var $coupon kDBItem */

		$coupon->Load($code, 'Code');

		if ( !$coupon->isLoaded() ) {
			$event->status = kEvent::erFAIL;
			$object->setCheckoutError(OrderCheckoutErrorType::COUPON, OrderCheckoutError::COUPON_CODE_INVALID);
			$event->redirect = false; // check!!!

			return ;
		}

		$expire_date = $coupon->GetDBField('Expiration');
		$number_of_use = $coupon->GetDBField('NumberOfUses');
		if ( $coupon->GetDBField('Status') != 1 || ($expire_date && $expire_date < adodb_mktime()) ||
			(isset($number_of_use) && $number_of_use <= 0))
		{
			$event->status = kEvent::erFAIL;
			$object->setCheckoutError(OrderCheckoutErrorType::COUPON, OrderCheckoutError::COUPON_CODE_EXPIRED);
			$event->redirect = false;

			return ;
		}

		$last_used = adodb_mktime();
		$coupon->SetDBField('LastUsedBy', $this->Application->RecallVar('user_id'));
		$coupon->SetDBField('LastUsedOn_date', $last_used);
		$coupon->SetDBField('LastUsedOn_time', $last_used);


		if ( isset($number_of_use) ) {
			$coupon->SetDBField('NumberOfUses', $number_of_use - 1);

			if ($number_of_use == 1) {
				$coupon->SetDBField('Status', 2);
			}
		}

		$coupon->Update();

		$this->Application->setUnitOption('ord', 'AutoLoad', true);
		$order = $this->Application->recallObject('ord');
		/* @var $order OrdersItem */

		$order->SetDBField('CouponId', $coupon->GetDBField('CouponId'));
		$order->SetDBField('CouponName', $coupon->GetDBField('Name')); // calculated field

		$order->Update();

		$object->setCheckoutError(OrderCheckoutErrorType::COUPON, OrderCheckoutError::COUPON_APPLIED);
//		OnApplyCoupon is called as hook for OnUpdateCart/OnCheckout, which calls OnRecalcualate themself
	}

	/**
	 * Removes coupon from order
	 *
	 * @param kEvent $event
	 * @deprecated
	 */
	function OnRemoveCoupon($event)
	{
		$object = $event->getObject();
		/* @var $object OrdersItem */

		$this->RemoveCoupon($object);
		$object->setCheckoutError(OrderCheckoutErrorType::COUPON, OrderCheckoutError::COUPON_REMOVED);

		$event->CallSubEvent('OnRecalculateItems');
	}

	/**
	 * Removes coupon from a given order
	 *
	 * @param OrdersItem $object
	 */
	function RemoveCoupon(&$object)
	{
		$coupon = $this->Application->recallObject('coup', null, Array('skip_autoload' => true));
		/* @var $coupon kDBItem */

		$coupon->Load( $object->GetDBField('CouponId') );

		if ( $coupon->isLoaded() ) {
			$coupon->SetDBField('NumberOfUses', $coupon->GetDBField('NumberOfUses') + 1);
			$coupon->SetDBField('Status', STATUS_ACTIVE);
			$coupon->Update();
		}

		$object->SetDBField('CouponId', 0);
		$object->SetDBField('CouponName', ''); // calculated field
		$object->SetDBField('CouponDiscount', 0);
	}

	/**
	 * Enter description here...
	 *
	 * @param kEvent $event
	 */
	function OnAddVirtualProductToCart($event)
	{
		$l_info = $this->Application->GetVar('l');
		if($l_info)
		{
			foreach($l_info as $link_id => $link_info) {}
			$item_data['LinkId'] = $link_id;
			$item_data['ListingTypeId'] = $link_info['ListingTypeId'];
		}
		else
		{
			$link_id = $this->Application->GetVar('l_id');
			$sql = 'SELECT ResourceId FROM '.$this->Application->getUnitOption('l', 'TableName').'
					WHERE LinkId = '.$link_id;
			$sql = 'SELECT ListingTypeId FROM '.$this->Application->getUnitOption('ls', 'TableName').'
					WHERE ItemResourceId = '.$this->Conn->GetOne($sql);
			$item_data['LinkId'] = $link_id;
			$item_data['ListingTypeId'] = $this->Conn->GetOne($sql);
		}

		$sql = 'SELECT VirtualProductId FROM '.$this->Application->getUnitOption('lst', 'TableName').'
				WHERE ListingTypeId = '.$item_data['ListingTypeId'];
		$item_id = $this->Conn->GetOne($sql);

		$event->setEventParam('ItemData', serialize($item_data));
		$this->AddItemToOrder($event, $item_id);

		$shop_cart_template = $this->Application->GetVar('shop_cart_template');

		if ( $shop_cart_template ) {
			$event->redirect = $shop_cart_template;
		}

		// don't pass unused info to shopping cart, brokes old mod-rewrites
		$event->SetRedirectParam('pass', 'm'); // not to pass link id
		$event->SetRedirectParam('m_cat_id', 0); // not to pass link id
	}

	function OnRemoveFromCart($event)
	{
		$ord_item_id = $this->Application->GetVar('orditems_id');
		$ord_id = $this->getPassedID($event);
		$this->Conn->Query('DELETE FROM '.TABLE_PREFIX.'OrderItems WHERE OrderId = '.$ord_id.' AND OrderItemId = '.$ord_item_id);
		$this->OnRecalculateItems($event);
	}

	function OnCleanupCart($event)
	{
		$object = $event->getObject();

		$sql = 'DELETE FROM '.TABLE_PREFIX.'OrderItems
				WHERE OrderId = '.$this->getPassedID($event);
		$this->Conn->Query($sql);

		$this->RemoveCoupon($object);
		$this->RemoveGiftCertificate($object);

		$this->OnRecalculateItems($event);
	}

	/**
	 * Returns order id from session or last used
	 *
	 * @param kEvent $event
	 * @return int
	 * @access public
	 */
	public function getPassedID(kEvent $event)
	{
		$event->setEventParam('raise_warnings', 0);
		$passed = parent::getPassedID($event);

		if ( $this->Application->isAdminUser ) {
			// work as usual in admin
			return $passed;
		}

		if ( $event->Special == 'last' ) {
			// return last order id (for using on thank you page)
			$order_id = $this->Application->RecallVar('front_order_id');

			return $order_id > 0 ? $order_id : FAKE_ORDER_ID; // FAKE_ORDER_ID helps to keep parent filter for order items set in "kDBList::linkToParent"
		}

		$ses_id = $this->Application->RecallVar($event->getPrefixSpecial(true) . '_id');

		if ( $passed && ($passed != $ses_id) ) {
			// order id given in url doesn't match our current order id
			$sql = 'SELECT PortalUserId
					FROM ' . TABLE_PREFIX . 'Orders
					WHERE OrderId = ' . $passed;
			$user_id = $this->Conn->GetOne($sql);

			if ( $user_id == $this->Application->RecallVar('user_id') ) {
				// current user is owner of order with given id -> allow him to view order details
				return $passed;
			}
			else {
				// current user is not owner of given order -> hacking attempt
				$this->Application->SetVar($event->getPrefixSpecial() . '_id', 0);
				return 0;
			}
		}

		// not passed or equals to ses_id
		return $ses_id > 0 ? $ses_id : FAKE_ORDER_ID; // FAKE_ORDER_ID helps to keep parent filter for order items set in "kDBList::linkToParent"
	}

	/**
	 * Load item if id is available
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function LoadItem(kEvent $event)
	{
		$id = $this->getPassedID($event);

		if ( $id == FAKE_ORDER_ID ) {
			// if we already know, that there is no such order,
			// then don't run database query, that will confirm that

			$object = $event->getObject();
			/* @var $object kDBItem */

			$object->Clear($id);
			return;
		}

		parent::LoadItem($event);
	}

	/**
	 * Creates new shopping cart
	 *
	 * @param kEvent $event
	 */
	function _createNewCart($event)
	{
		$object = $event->getObject( Array('skip_autoload' => true) );
		/* @var $object kDBItem */

		$this->setNextOrderNumber($event);
		$object->SetDBField('Status', ORDER_STATUS_INCOMPLETE);
		$object->SetDBField('VisitId', $this->Application->RecallVar('visit_id') );

		// get user
		if ( $this->Application->LoggedIn() ) {
			$user = $this->Application->recallObject('u.current');
			/* @var $user UsersItem */

			$user_id = $user->GetID();
			$object->SetDBField('BillingEmail', $user->GetDBField('Email'));
		}
		else {
			$user_id = USER_GUEST;
		}

		$object->SetDBField('PortalUserId', $user_id);

		// get affiliate
		$affiliate_id = $this->isAffiliate($user_id);
		if ( $affiliate_id ) {
			$object->SetDBField('AffiliateId', $affiliate_id);
		}
		else {
			$affiliate_storage_method = $this->Application->ConfigValue('Comm_AffiliateStorageMethod');

			if ( $affiliate_storage_method == 1 ) {
				$object->SetDBField('AffiliateId', (int)$this->Application->RecallVar('affiliate_id'));
			}
			else {
				$object->SetDBField('AffiliateId', (int)$this->Application->GetVar('affiliate_id'));
			}
		}

		// get payment type
		$default_type = $this->_getDefaultPaymentType();

		if ( $default_type ) {
			$object->SetDBField('PaymentType', $default_type);
		}

		// vat setting
		$object->SetDBField('VATIncluded', $this->Application->ConfigValue('OrderVATIncluded'));

		$created = $object->Create();

		if ( $created ) {
			$id = $object->GetID();

			$this->Application->SetVar($event->getPrefixSpecial(true) . '_id', $id);
			$this->Application->StoreVar($event->getPrefixSpecial(true) . '_id', $id);
			$this->Application->Session->SetCookie('shop_cart_cookie', $id, strtotime('+1 month'));

			return $id;
		}

		return 0;
	}

	/**
	 * Returns default payment type for order
	 *
	 * @return int
	 */
	function _getDefaultPaymentType()
	{
		$default_type = $this->Application->siteDomainField('PrimaryPaymentTypeId');

		if (!$default_type) {
			$sql = 'SELECT PaymentTypeId
					FROM ' . TABLE_PREFIX . 'PaymentTypes
					WHERE IsPrimary = 1';
			$default_type = $this->Conn->GetOne($sql);
		}

		return $default_type;
	}

	function StoreContinueShoppingLink()
	{
		$this->Application->StoreVar('continue_shopping', 'external:'.PROTOCOL.SERVER_NAME.$this->Application->RecallVar('last_url'));
	}

	/**
	 * Sets required fields for order, based on current checkout step
	 * !!! Do not use switch here, since all cases may be on the same form simultaneously
	 *
	 * @param kEvent $event
	 */
	function SetStepRequiredFields($event)
	{
		$order = $event->getObject();
		/* @var $order OrdersItem */

		$cs_helper = $this->Application->recallObject('CountryStatesHelper');
		/* @var $cs_helper kCountryStatesHelper */

		$items_info = $this->Application->GetVar($event->getPrefixSpecial(true));
		if ($items_info) {
			// updated address available from SUBMIT -> use it
			list($id, $field_values) = each($items_info);
		}
		else {
			// no updated address -> use current address
			$field_values = Array (
				'ShippingCountry' => $order->GetDBField('ShippingCountry'),
				'BillingCountry' => $order->GetDBField('BillingCountry'),
				'PaymentType' => $order->GetDBField('PaymentType'),
			);
		}

		// shipping address required fields
		if ($this->Application->GetVar('check_shipping_address')) {
			$has_tangibles = $order->HasTangibleItems();
			$req_fields = array('ShippingTo', 'ShippingAddress1', 'ShippingCity', 'ShippingZip', 'ShippingCountry', /*'ShippingPhone',*/ 'BillingEmail');
			$order->setRequired($req_fields, $has_tangibles);
			$order->setRequired('ShippingState', $cs_helper->CountryHasStates( $field_values['ShippingCountry'] ));
		}

		// billing address required fields
		if ($this->Application->GetVar('check_billing_address')) {
			$req_fields = array('BillingTo', 'BillingAddress1', 'BillingCity', 'BillingZip', 'BillingCountry', 'BillingPhone', 'BillingEmail');
			$order->setRequired($req_fields);
			$order->setRequired('BillingState', $cs_helper->CountryHasStates( $field_values['BillingCountry'] ));
		}

		$check_cc = $this->Application->GetVar('check_credit_card');

		if ( $check_cc && ($field_values['PaymentType'] == $order->GetDBField('PaymentType')) ) {
			// cc check required AND payment type was not changed during SUBMIT
			if ( $this->Application->isAdminUser ) {
				$req_fields = Array (/*'PaymentCardType',*/ 'PaymentAccount', /*'PaymentNameOnCard',*/ 'PaymentCCExpDate');
			}
			else {
				$req_fields = Array (/*'PaymentCardType',*/ 'PaymentAccount', /*'PaymentNameOnCard',*/ 'PaymentCCExpDate', 'PaymentCVV2');
			}

			$order->setRequired($req_fields);
		}
	}

	/**
	 * Set's order's user_id to user from session or Guest otherwise
	 *
	 * @param kEvent $event
	 */
	function CheckUser($event)
	{
		if ( $this->Application->isAdminUser || defined('GW_NOTIFY') || defined('CRON') ) {
			// 1. don't check, when Administrator is editing the order.
			// 2. don't check while processing payment gateways, because they can do cross-domain ssl redirects.
			// 3. don't check from CRON, because it's like Admin updates orders on other user behalf.
			return;
		}

		$order = $event->getObject();
		/* @var $order OrdersItem */

		$ses_user = $this->Application->RecallVar('user_id');

		if ( $order->GetDBField('PortalUserId') != $ses_user ) {
			if ( $ses_user == 0 ) {
				$ses_user = USER_GUEST;
			}

			$order->SetDBField('PortalUserId', $ses_user);
			// since CheckUser is called in OnBeforeItemUpdate, we don't need to call udpate here, just set the field
		}
	}

	/* ======================== ADMIN ONLY ======================== */

	/**
	 * Prepare temp tables for creating new item
	 * but does not create it. Actual create is
	 * done in OnPreSaveCreated
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function OnPreCreate(kEvent $event)
	{
		parent::OnPreCreate($event);

		$object = $event->getObject();
		/* @var $object kDBItem */

		$this->setNextOrderNumber($event);

		$object->SetDBField('OrderIP', $this->Application->getClientIp());

		$order_type = $this->getTypeBySpecial( $this->Application->GetVar('order_type') );
		$object->SetDBField('Status', $order_type);
	}

	/**
	 * When cloning orders set new order number to them
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function OnBeforeClone(kEvent $event)
	{
		parent::OnBeforeClone($event);

		/** @var OrdersItem $object */
		$object = $event->getObject();

		if ( substr($event->Special, 0, 9) == 'recurring' ) {
			$object->SetDBField('SubNumber', $object->getNextSubNumber());
		}
		else {
			$this->setNextOrderNumber($event);
		}

		$reset_fields = array(
			'OnHold', 'OrderDate', 'ShippingCost', 'ShippingTracking', 'ShippingDate', 'ReturnTotal',
			'OriginalAmount', 'ShippingInfo', 'GWResult1', 'GWResult2', 'AffiliateCommission',
			'ProcessingFee', 'InsuranceFee',
		);

		foreach ( $reset_fields as $reset_field ) {
			$field_options = $object->GetFieldOptions($reset_field);
			$object->SetDBField($reset_field, $field_options['default']);
		}

		$object->SetDBField('OrderIP', $this->Application->getClientIp());
		$object->UpdateFormattersSubFields();
	}

	function OnReserveItems($event)
	{
		$order_items = $this->Application->recallObject('orditems.-inv','orditems_List',Array('skip_counting'=>true,'per_page'=>-1) );
		/* @var $order_items kDBList */

		$order_items->linkToParent('-inv');
		// force re-query, since we are updateing through orditem ITEM, not the list, and
		// OnReserveItems may be called 2 times when fullfilling backorders through product edit - first time
		// from FullFillBackorders and second time from OnOrderProcess
		$order_items->Query(true);
		$order_items->GoFirst();

		// query all combinations used in this order


		$product_object = $this->Application->recallObject('p', null, Array('skip_autoload' => true));
		/* @var $product_object kCatDBItem */

		$product_object->SwitchToLive();

		$order_item = $this->Application->recallObject('orditems.-item', null, Array('skip_autoload' => true));
		/* @var $order_item kDBItem */

		$combination_item = $this->Application->recallObject('poc.-item', null, Array('skip_autoload' => true));
		/* @var $combination_item kDBItem */

		$combinations = $this->queryCombinations($order_items);

		$event->status = kEvent::erSUCCESS;
		while (!$order_items->EOL()) {
			$rec = $order_items->getCurrentRecord();
			$product_object->Load( $rec['ProductId'] );
			if (!$product_object->GetDBField('InventoryStatus')) {
				$order_items->GoNext();
				continue;
			}

			$inv_object =& $this->getInventoryObject($product_object, $combination_item, $combinations[ $rec['ProductId'].'_'.$rec['OptionsSalt'] ]);

			$lack = $rec['Quantity'] - $rec['QuantityReserved'];

			if ( $lack > 0 ) {
				// Reserve lack or what is available (in case if we need to reserve anything, by Alex).
				$to_reserve = min(
					$lack,
					$inv_object->GetDBField('QtyInStock') - $product_object->GetDBField('QtyInStockMin')
				);

				// If we can't reserve the full lack.
				if ( $to_reserve < $lack ) {
					$event->status = kEvent::erFAIL;
				}

				// Reserve in order.
				$order_item->SetDBFieldsFromHash($rec);
				$order_item->SetDBField('QuantityReserved', $rec['QuantityReserved'] + $to_reserve);
				$new_lack = $order_item->GetDBField('Quantity') - $order_item->GetDBField('QuantityReserved');
				$order_item->SetDBField('BackOrderFlag', abs($new_lack) <= 0.0001 ? 0 : 1);
				$order_item->SetId($rec['OrderItemId']);
				$order_item->Update();

				// Update product - increase reserved, decrease in stock.
				$inv_object->SetDBField('QtyReserved', $inv_object->GetDBField('QtyReserved') + $to_reserve);
				$inv_object->SetDBField('QtyInStock', $inv_object->GetDBField('QtyInStock') - $to_reserve);
				$inv_object->SetDBField('QtyBackOrdered', $inv_object->GetDBField('QtyBackOrdered') - $to_reserve);
				$inv_object->Update();

				if ( $product_object->GetDBField('InventoryStatus') == 2 ) {
					// Inventory by options, then restore changed combination
					// values back to common $combinations array !!!
					$combinations[$rec['ProductId'] . '_' . $rec['OptionsSalt']] = $inv_object->GetFieldValues();
				}
			}
			$order_items->GoNext();
		}
		return true;
	}

	function OnOrderPrint($event)
	{
		$event->SetRedirectParam('opener', 's');
	}

	/**
	 * Processes order each tab info resetting to other tab info / to user info
	 *
	 * @param kEvent $event
	 * @access public
	 */
	function OnResetAddress($event)
	{
		$to_tab = $this->Application->GetVar('to_tab');
		$from_tab = substr($event->Name, strlen('OnResetTo'));

		// load values from db
		$object = $event->getObject();
		/* @var $object kDBItem */

		// update values from submit
		$field_values = $this->getSubmittedFields($event);
		$object->SetFieldsFromHash($field_values);
		$event->setEventParam('form_data', $field_values);

		$this->DoResetAddress($object, $from_tab, $to_tab);

		$object->Update();
		$event->redirect = false;
	}

	/**
	 * Processes item selection from popup item selector
	 *
	 * @todo Is this called ? (by Alex)
	 * @param kEvent $event
	 */
	function OnProcessSelected($event)
	{
		$selected_ids = $this->Application->GetVar('selected_ids');
		$product_ids = $selected_ids['p'];

		if ($product_ids) {
			$product_ids = explode(',', $product_ids);

			// !!! LOOK OUT - Adding items to Order in admin is handled in order_ITEMS_event_handler !!!
			foreach ($product_ids as $product_id) {
				$this->AddItemToOrder($event, $product_id);
			}
		}

		$event->SetRedirectParam('opener', 'u');
	}

	function OnMassPlaceOrder($event)
	{
		$object = $event->getObject( Array('skip_autoload' => true) );
		$ids = $this->StoreSelectedIDs($event);

		if($ids)
		{
			foreach($ids as $id)
			{
				$object->Load($id);
				$this->DoPlaceOrder($event);
			}
		}
		$event->status = kEvent::erSUCCESS;

	}



	/**
	 * Universal
	 * Checks if QtyInStock is enough to fullfill backorder (Qty - QtyReserved in order)
	 *
	 * @param int $ord_id
	 * @return bool
	 */
	function ReadyToProcess($ord_id)
	{
		$poc_table = $this->Application->getUnitOption('poc', 'TableName');
		$query = '	SELECT SUM(IF( IF('.TABLE_PREFIX.'Products.InventoryStatus = 2, '.$poc_table.'.QtyInStock, '.TABLE_PREFIX.'Products.QtyInStock) - '.TABLE_PREFIX.'Products.QtyInStockMin >= ('.TABLE_PREFIX.'OrderItems.Quantity - '.TABLE_PREFIX.'OrderItems.QuantityReserved), 0, 1))
							FROM '.TABLE_PREFIX.'OrderItems
					LEFT JOIN '.TABLE_PREFIX.'Products ON '.TABLE_PREFIX.'Products.ProductId = '.TABLE_PREFIX.'OrderItems.ProductId
					LEFT JOIN '.$poc_table.' ON ('.$poc_table.'.CombinationCRC = '.TABLE_PREFIX.'OrderItems.OptionsSalt) AND ('.$poc_table.'.ProductId = '.TABLE_PREFIX.'OrderItems.ProductId)
							WHERE OrderId = '.$ord_id.'
							GROUP BY OrderId';

		// IF (IF(InventoryStatus = 2, poc.QtyInStock, p.QtyInStock) - QtyInStockMin >= (Quantity - QuantityReserved), 0, 1
		return ($this->Conn->GetOne($query) == 0);
	}

	/**
	 * Return all option combinations used in order
	 *
	 * @param kDBList $order_items
	 * @return Array
	 */
	function queryCombinations(&$order_items)
	{
		// 1. collect combination crc used in order
		$combinations = Array();
		while (!$order_items->EOL()) {
			$row = $order_items->getCurrentRecord();
			if ($row['OptionsSalt'] == 0) {
				$order_items->GoNext();
				continue;
			}
			$combinations[] = '(poc.ProductId = '.$row['ProductId'].') AND (poc.CombinationCRC = '.$row['OptionsSalt'].')';
			$order_items->GoNext();
		}
		$order_items->GoFirst();
		$combinations = array_unique($combinations); // if same combination+product found as backorder & normal order item

		if ($combinations) {
			// 2. query data about combinations
			$poc_table = $this->Application->getUnitOption('poc', 'TableName');
			$sql = 'SELECT CONCAT(poc.ProductId, "_", poc.CombinationCRC) AS CombinationKey, poc.*
					FROM '.$poc_table.' poc
					WHERE ('.implode(') OR (', $combinations).')';

			return $this->Conn->Query($sql, 'CombinationKey');
		}

		return Array();
	}

	/**
	 * Returns object to perform inventory actions on
	 *
	 * @param ProductsItem $product current product object in order
	 * @param kDBItem $combination combination dummy object
	 * @param Array $combination_data pre-queried combination data
	 * @return kDBItem
	 */
	function &getInventoryObject(&$product, &$combination, $combination_data)
	{
		if ($product->GetDBField('InventoryStatus') == 2) {
			// inventory by option combinations
			$combination->SetDBFieldsFromHash($combination_data);
			$combination->setID($combination_data['CombinationId']);
			$change_item =& $combination;
		}
		else {
			// inventory by product ifself
			$change_item =& $product;
		}

		return $change_item;
	}

	/**
	 * Approve order ("Pending" tab)
	 *
	 * @param kDBList $order_items
	 * @return int new status of order if any
	 */
	function approveOrder(&$order_items)
	{
		$product_object = $this->Application->recallObject('p', null, Array('skip_autoload' => true));
		$order_item = $this->Application->recallObject('orditems.-item', null, Array('skip_autoload' => true));
		$combination_item = $this->Application->recallObject('poc.-item', null, Array('skip_autoload' => true));

		$combinations = $this->queryCombinations($order_items);

		while (!$order_items->EOL()) {
			$rec = $order_items->getCurrentRecord();

			$order_item->SetDBFieldsFromHash($rec);
			$order_item->SetId($rec['OrderItemId']);
			$order_item->SetDBField('QuantityReserved', 0);
			$order_item->Update();

			$product_object->Load( $rec['ProductId'] );
			if (!$product_object->GetDBField('InventoryStatus')) {
				// if no inventory info is collected, then skip this order item
				$order_items->GoNext();
				continue;
			}

			$inv_object =& $this->getInventoryObject($product_object, $combination_item, $combinations[ $rec['ProductId'].'_'.$rec['OptionsSalt'] ]);

			// decrease QtyReserved by amount of product used in order
			$inv_object->SetDBField('QtyReserved', $inv_object->GetDBField('QtyReserved') - $rec['Quantity']);
			$inv_object->Update();

			if ($product_object->GetDBField('InventoryStatus') == 2) {
				// inventory by options, then restore changed combination values back to common $combinations array !!!
				$combinations[ $rec['ProductId'].'_'.$rec['OptionsSalt'] ] = $inv_object->GetFieldValues();
			}

			$order_items->GoNext();
		}
		return true;
	}

	/**
	 * Restores reserved items in the order
	 *
	 * @param kDBList $order_items
	 * @return bool
	 */
	function restoreOrder(&$order_items)
	{
		$product_object = $this->Application->recallObject('p', null, Array('skip_autoload' => true));
		/* @var $product_object kCatDBItem */

		$product_object->SwitchToLive();

		$order_item = $this->Application->recallObject('orditems.-item', null, Array('skip_autoload' => true));
		/* @var $order_item kDBItem */

		$combination_item = $this->Application->recallObject('poc.-item', null, Array('skip_autoload' => true));
		/* @var $combination_item kDBItem */

		$combinations = $this->queryCombinations($order_items);

		while( !$order_items->EOL() )
		{
			$rec = $order_items->getCurrentRecord();

			$product_object->Load( $rec['ProductId'] );
			if (!$product_object->GetDBField('InventoryStatus')) {
				// if no inventory info is collected, then skip this order item
				$order_items->GoNext();
				continue;
			}

			$inv_object =& $this->getInventoryObject($product_object, $combination_item, $combinations[ $rec['ProductId'].'_'.$rec['OptionsSalt'] ]);

			// cancelling backorderd qty if any
			$lack = $rec['Quantity'] - $rec['QuantityReserved'];
			if ($lack > 0 && $rec['BackOrderFlag'] > 0) { // lack should have been recorded as QtyBackOrdered
				$inv_object->SetDBField('QtyBackOrdered', $inv_object->GetDBField('QtyBackOrdered') - $lack);
			}

			// canceling reservation in stock
			$inv_object->SetDBField('QtyReserved', $inv_object->GetDBField('QtyReserved') - $rec['QuantityReserved']);
			// putting remaining freed qty back to stock
			$inv_object->SetDBField('QtyInStock', $inv_object->GetDBField('QtyInStock') + $rec['QuantityReserved']);
			$inv_object->Update();

			$product_h = $this->Application->recallObject('p_EventHandler');
			/* @var $product_h ProductsEventHandler */

			if ($product_object->GetDBField('InventoryStatus') == 2) {
				// inventory by options, then restore changed combination values back to common $combinations array !!!
				$combinations[ $rec['ProductId'].'_'.$rec['OptionsSalt'] ] = $inv_object->GetFieldValues();

				// using freed qty to fulfill possible backorders
				$product_h->FullfillBackOrders($product_object, $inv_object->GetID());
			}
			else {
				// using freed qty to fulfill possible backorders
				$product_h->FullfillBackOrders($product_object, 0);
			}

			$order_item->SetDBFieldsFromHash($rec);
			$order_item->SetId($rec['OrderItemId']);
			$order_item->SetDBField('QuantityReserved', 0);
			$order_item->Update();

			$order_items->GoNext();
		}

		return true;
	}

	/**
	 * Approve order + special processing
	 *
	 * @param kEvent $event
	 */
	function MassInventoryAction($event)
	{
		if ( $this->Application->CheckPermission('SYSTEM_ACCESS.READONLY', 1) ) {
			$event->status = kEvent::erFAIL;
			return;
		}

		// process order products
		$object = $this->Application->recallObject($event->Prefix . '.-inv', null, Array ('skip_autoload' => true));
		/* @var $object kDBItem */

		$ids = $this->StoreSelectedIDs($event);

		if ( $ids ) {
			foreach ($ids as $id) {
				$object->Load($id);
				$this->InventoryAction($event);
			}
		}
	}

	function InventoryAction($event)
	{
		if ($this->Application->CheckPermission('SYSTEM_ACCESS.READONLY', 1)) {
			$event->status = kEvent::erFAIL;
			return;
		}

		$event_status_map = Array(
			'OnMassOrderApprove'	=> ORDER_STATUS_TOSHIP,
			'OnOrderApprove' 		=> ORDER_STATUS_TOSHIP,
			'OnMassOrderDeny'		=> ORDER_STATUS_DENIED,
			'OnOrderDeny'			=> ORDER_STATUS_DENIED,
			'OnMassOrderArchive'	=> ORDER_STATUS_ARCHIVED,
			'OnOrderArchive'		=> ORDER_STATUS_ARCHIVED,
			'OnMassOrderShip' 		=> ORDER_STATUS_PROCESSED,
			'OnOrderShip' 			=> ORDER_STATUS_PROCESSED,
			'OnMassOrderProcess' 	=> ORDER_STATUS_TOSHIP,
			'OnOrderProcess' 		=> ORDER_STATUS_TOSHIP,
		);

		$order_items = $this->Application->recallObject('orditems.-inv','orditems_List',Array('skip_counting'=>true,'per_page'=>-1) );
		/* @var $order_items kDBList */

		$order_items->linkToParent('-inv');
		$order_items->Query();
		$order_items->GoFirst();

		$object = $this->Application->recallObject($event->Prefix.'.-inv');
		/* @var $object OrdersItem */

		if ($object->GetDBField('OnHold')) {
			// any actions have no effect while on hold
			return ;
		}

		// save original order status
		$original_order_status = $object->GetDBField('Status');

		// preparing new status, but not setting it yet
		$object->SetDBField('Status', $event_status_map[$event->Name]);

		$set_new_status = false;
		$event->status = kEvent::erSUCCESS;

		$email_params = $this->OrderEmailParams($object);

		switch ($event->Name) {
			case 'OnMassOrderApprove':
			case 'OnOrderApprove':
				$set_new_status = false; //on successful approve order will be split and new orders will have new statuses

				if ($object->GetDBField('ChargeOnNextApprove')) {
					$charge_info = $this->ChargeOrder($object);
					if (!$charge_info['result']) {
						break;
					}

					// removing ChargeOnNextApprove
					$object->SetDBField('ChargeOnNextApprove', 0);
					$sql = 'UPDATE '.$object->TableName.' SET ChargeOnNextApprove = 0 WHERE '.$object->IDField.' = '.$object->GetID();
					$this->Conn->Query($sql);
				}

				// charge user for order in case if we user 2step charging (e.g. AUTH_ONLY + PRIOR_AUTH_CAPTURE)
				$gw_data = $object->getGatewayData();

				/** @var kGWBase $gateway_object */
				$gateway_object = $this->Application->recallObject($gw_data['ClassName']);

				$charge_result = $gateway_object->Charge($object->GetFieldValues(), $gw_data['gw_params']);
				$sql = 'UPDATE %s SET GWResult2 = %s WHERE %s = %s';
				$sql = sprintf($sql, $object->TableName, $this->Conn->qstr($gateway_object->getGWResponce()), $object->IDField, $object->GetID() );
				$this->Conn->Query($sql);
				$object->SetDBField('GWResult2', $gateway_object->getGWResponce() );

				if ($charge_result) {
					$product_object = $this->Application->recallObject('p', null, Array('skip_autoload' => true));
					/* @var $product_object ProductsItem */

					foreach ($order_items->Records as $product_item) {
						if (!$product_item['ProductId']) {
							 // product may have been deleted
							continue;
						}
						$product_object->Load($product_item['ProductId']);
						$hits = floor( $product_object->GetDBField('Hits') ) + 1;
						$sql = 'SELECT MAX(Hits) FROM '.$this->Application->getUnitOption('p', 'TableName').'
								WHERE FLOOR(Hits) = '.$hits;
						$hits = ( $res = $this->Conn->GetOne($sql) ) ? $res + 0.000001 : $hits;
						$product_object->SetDBField('Hits', $hits);
						$product_object->Update();

						/*$sql = 'UPDATE '.$this->Application->getUnitOption('p', 'TableName').'
								SET Hits = Hits + '.$product_item['Quantity'].'
								WHERE ProductId = '.$product_item['ProductId'];
						$this->Conn->Query($sql);*/
					}

					$this->PrepareCoupons($event, $object);
					$this->SplitOrder($event, $object);

					if ( $object->GetDBField('IsRecurringBilling') != 1 ) {
						$this->Application->emailUser('ORDER.APPROVE', null, $email_params);

						// Mask credit card with XXXX
						if ( $this->Application->ConfigValue('Comm_MaskProcessedCreditCards') ) {
							$this->maskCreditCard($object, 'PaymentAccount');
							$set_new_status = 1;
						}
					}
				}

				break;

			case 'OnMassOrderDeny':
			case 'OnOrderDeny':
				foreach ($order_items->Records as $product_item) {
					if (!$product_item['ProductId']) {
						 // product may have been deleted
						continue;
					}
					$this->raiseProductEvent('Deny', $product_item['ProductId'], $product_item);
				}

				if ( ($original_order_status != ORDER_STATUS_INCOMPLETE) && ($event->Name == 'OnMassOrderDeny' || $event->Name == 'OnOrderDeny') ) {
					$this->Application->emailUser('ORDER.DENY', null, $email_params);

					// inform payment gateway that order was declined
					$gw_data = $object->getGatewayData();

					if ( $gw_data ) {
						/** @var kGWBase $gateway_object */
						$gateway_object = $this->Application->recallObject($gw_data['ClassName']);
						$gateway_object->OrderDeclined($object->GetFieldValues(), $gw_data['gw_params']);
					}
				}

				// !!! LOOK HERE !!!
				// !!!! no break !!!! here on purpose!!!
			case 'OnMassOrderArchive':
			case 'OnOrderArchive':
				// it's critical to update status BEFORE processing items because
				// FullfillBackorders could be called during processing and in case
				// of order denial/archive fullfill could reserve the qtys back for current backorder
				$object->Update();
				$this->restoreOrder($order_items);
				$set_new_status = false; // already set
				break;

			case 'OnMassOrderShip':
			case 'OnOrderShip':
				$ret = Array ();
				$shipping_info = $object->GetDBField('ShippingInfo');

				if ($shipping_info) {
					$quote_engine_collector = $this->Application->recallObject('ShippingQuoteCollector');
					/* @var $quote_engine_collector ShippingQuoteCollector */

					$shipping_info = unserialize($shipping_info);
					$sqe_class_name = $quote_engine_collector->GetClassByType($shipping_info, 1);
				}

				// try to create usps order
				if (($object->GetDBField('ShippingType') == 0) && ($sqe_class_name !== false)) {
					$shipping_quote_engine = $this->Application->recallObject($sqe_class_name);
					/* @var $shipping_quote_engine ShippingQuoteEngine */

					$ret = $shipping_quote_engine->MakeOrder($object);
				}

				if ( !array_key_exists('error_number', $ret) ) {
					$set_new_status = $this->approveOrder($order_items);

//					$set_new_status = $this->shipOrder($order_items);
					$object->SetDBField('ShippingDate', adodb_mktime());
					$object->UpdateFormattersSubFields();

					$shipping_email = $object->GetDBField('ShippingEmail');
					$email_params['to_email'] = $shipping_email ? $shipping_email : $email_params['_user_email'];
					$this->Application->emailUser('ORDER.SHIP', null, $email_params);

					// inform payment gateway that order was shipped
					$gw_data = $object->getGatewayData();

					/** @var kGWBase $gateway_object */
					$gateway_object = $this->Application->recallObject($gw_data['ClassName']);
					$gateway_object->OrderShipped($object->GetFieldValues(), $gw_data['gw_params']);
				}
				else {
					$sqe_errors = $this->Application->RecallVar('sqe_errors');
					$sqe_errors = $sqe_errors ? unserialize($sqe_errors) : Array ();
					$sqe_errors[ $object->GetField('OrderNumber') ] = $ret['error_description'];

					$this->Application->StoreVar('sqe_errors', serialize($sqe_errors));
				}
				break;

			case 'OnMassOrderProcess':
			case 'OnOrderProcess':
				if ( $this->ReadyToProcess($object->GetID()) ) {
					$event->CallSubEvent('OnReserveItems');

					if ( $event->status == kEvent::erSUCCESS ) {
						$set_new_status = true;
					}

					$this->Application->emailUser('BACKORDER.PROCESS', null, $email_params);
				}
				else {
					$event->status = kEvent::erFAIL;
				}
				break;
		}

		if ( $set_new_status ) {
			$object->Update();
		}
	}

	/**
	 * Hides last 4 digits from credit card number
	 *
	 * @param OrdersItem $object
	 * @param string $field
	 */
	function maskCreditCard(&$object, $field)
	{
		$value = $object->GetDBField($field);
		$value = preg_replace('/'.substr($value, -4).'$/', str_repeat('X', 4), $value);
		$object->SetDBField($field, $value);
	}

	/**
	 * Set next available order number.
	 *
	 * @param kEvent $event Event.
	 *
	 * @return void
	 */
	protected function setNextOrderNumber(kEvent $event)
	{
		/** @var OrdersItem $object */
		$object = $event->getObject();

		$next_order_number = $this->getNextOrderNumber();

		$object->SetDBField('Number', $next_order_number);
		$object->SetDBField('SubNumber', 0);

		// set virtual field too
		$number_format = (int)$this->Application->ConfigValue('Comm_Order_Number_Format_P');
		$sub_number_format = (int)$this->Application->ConfigValue('Comm_Order_Number_Format_S');
		$order_number = sprintf('%0' . $number_format . 'd', $next_order_number) . '-' . str_repeat('0', $sub_number_format);

		$object->SetDBField('OrderNumber', $order_number);
	}

	/**
	 * Returns order number to be used next.
	 *
	 * @return integer
	 */
	protected function getNextOrderNumber()
	{
		$config_table = $this->Application->getUnitOption('conf', 'TableName');
		$this->Conn->Query('LOCK TABLES ' . $config_table . ' WRITE');

		$sql = 'UPDATE ' . $config_table . '
				SET VariableValue = VariableValue + 1
				WHERE VariableName = "Comm_Next_Order_Number"';
		$this->Conn->Query($sql);

		$sql = 'SELECT VariableValue
				FROM ' . $config_table . '
				WHERE VariableName = "Comm_Next_Order_Number"';
		$next_order_number = $this->Conn->GetOne($sql);

		$this->Conn->Query('UNLOCK TABLES');
		$this->Application->SetConfigValue('Comm_Next_Order_Number', $next_order_number);

		return $next_order_number - 1;
	}

	/**
	 * [HOOK] Ensures, that "Next Order Number" system setting is within allowed limits.
	 *
	 * @param kEvent $event Event.
	 *
	 * @return void
	 */
	protected function OnBeforeNextOrderNumberChange(kEvent $event)
	{
		/** @var kDBItem $system_setting */
		$system_setting = $event->MasterEvent->getObject();

		$old_value = $system_setting->GetOriginalField('VariableValue');
		$new_value = $system_setting->GetDBField('VariableValue');

		if ( $system_setting->GetDBField('VariableName') != 'Comm_Next_Order_Number' || $new_value == $old_value ) {
			return;
		}

		$sql = 'SELECT MAX(Number)
				FROM ' . $this->Application->getUnitOption($event->Prefix, 'TableName');
		$next_order_number = (int)$this->Conn->GetOne($sql) + 1;

		if ( $new_value < $next_order_number ) {
			$system_setting->SetError('VariableValue', 'value_out_of_range', null, array(
				'min_value' => $next_order_number,
				'max_value' => '&infin;',
			));
		}
	}

	/**
	 * Set's new order address based on another address from order (e.g. billing from shipping)
	 *
	 * @param unknown_type $object
	 * @param unknown_type $from
	 * @param unknown_type $to
	 */
	function DoResetAddress(&$object, $from, $to)
	{
		$fields = Array('To','Company','Phone','Fax','Email','Address1','Address2','City','State','Zip','Country');

		if ($from == 'User') {
			// skip these fields when coping from user, because they are not present in user profile
			$tmp_fields = array_flip($fields);
//			unset($tmp_fields['Company'], $tmp_fields['Fax'], $tmp_fields['Address2']);
			$fields = array_flip($tmp_fields);
		}

		// apply modification
		foreach ($fields as $field_name) {
			$object->SetDBField($to.$field_name, $object->GetDBField($from.$field_name));
		}
	}

	/**
	 * Set's status incomplete to all cloned orders
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function OnAfterClone(kEvent $event)
	{
		parent::OnAfterClone($event);

		$id = $event->getEventParam('id');
		$table = $this->Application->getUnitOption($event->Prefix, 'TableName');
		$id_field = $this->Application->getUnitOption($event->Prefix, 'IDField');

		// set cloned order status to Incomplete
		$sql = 'UPDATE ' . $table . '
				SET Status = 0
				WHERE ' . $id_field . ' = ' . $id;
		$this->Conn->Query($sql);
	}


	/* ======================== COMMON CODE ======================== */

	/**
	 * Split one timestamp field into 2 virtual fields
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function OnAfterItemLoad(kEvent $event)
	{
		parent::OnAfterItemLoad($event);

		$object = $event->getObject();
		/* @var $object kDBItem */

		// get user fields
		$user_id = $object->GetDBField('PortalUserId');

		if ( $user_id ) {
			$sql = 'SELECT *, CONCAT(FirstName,\' \',LastName) AS UserTo
					FROM ' . TABLE_PREFIX . 'Users
					WHERE PortalUserId = ' . $user_id;
			$user_info = $this->Conn->GetRow($sql);

			$fields = Array(
				'UserTo'=>'UserTo','UserPhone'=>'Phone','UserFax'=>'Fax','UserEmail'=>'Email',
				'UserAddress1'=>'Street','UserAddress2'=>'Street2','UserCity'=>'City','UserState'=>'State',
				'UserZip'=>'Zip','UserCountry'=>'Country','UserCompany'=>'Company'
			);

			foreach ($fields as $object_field => $user_field) {
				$object->SetDBField($object_field, $user_info[$user_field]);
			}
		}

		$object->SetDBField('PaymentCVV2', $this->Application->RecallVar('CVV2Code'));

		$cs_helper = $this->Application->recallObject('CountryStatesHelper');
		/* @var $cs_helper kCountryStatesHelper */

		$cs_helper->PopulateStates($event, 'ShippingState', 'ShippingCountry');
		$cs_helper->PopulateStates($event, 'BillingState', 'BillingCountry');

		$this->SetStepRequiredFields($event);

		// needed in OnAfterItemUpdate
		$this->Application->SetVar('OriginalShippingOption', $object->GetDBField('ShippingOption'));
	}

	/**
	 * Processes states
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function OnBeforeItemCreate(kEvent $event)
	{
		parent::OnBeforeItemCreate($event);

		$cs_helper = $this->Application->recallObject('CountryStatesHelper');
		/* @var $cs_helper kCountryStatesHelper */

		$cs_helper->PopulateStates($event, 'ShippingState', 'ShippingCountry');
		$cs_helper->PopulateStates($event, 'BillingState', 'BillingCountry');
	}

	/**
	 * Processes states
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function OnBeforeItemUpdate(kEvent $event)
	{
		parent::OnBeforeItemUpdate($event);

		$object = $event->getObject();
		/* @var $object OrdersItem */

		$old_payment_type = $object->GetOriginalField('PaymentType');
		$new_payment_type = $object->GetDBField('PaymentType');

		if ( $new_payment_type != $old_payment_type ) {
			// payment type changed -> check that it's allowed
			$available_payment_types = $this->Application->siteDomainField('PaymentTypes');

			if ( $available_payment_types ) {
				if ( strpos($available_payment_types, '|' . $new_payment_type . '|') === false ) {
					// payment type isn't allowed in site domain
					$object->SetDBField('PaymentType', $old_payment_type);
				}
			}
		}

		$cs_helper = $this->Application->recallObject('CountryStatesHelper');
		/* @var $cs_helper kCountryStatesHelper */

		$cs_helper->PopulateStates($event, 'ShippingState', 'ShippingCountry');
		$cs_helper->PopulateStates($event, 'BillingState', 'BillingCountry');

		if ( $object->HasTangibleItems() ) {
			$cs_helper->CheckStateField($event, 'ShippingState', 'ShippingCountry', false);
		}

		$cs_helper->CheckStateField($event, 'BillingState', 'BillingCountry', false);

		if ( $object->GetDBField('Status') > ORDER_STATUS_PENDING ) {
			return ;
		}

		$this->CheckUser($event);

		if ( !$object->GetDBField('OrderIP') ) {
			$object->SetDBField('OrderIP', $this->Application->getClientIp());
		}

		$shipping_option = $this->Application->GetVar('OriginalShippingOption');
		$new_shipping_option = $object->GetDBField('ShippingOption');

		if ( $shipping_option != $new_shipping_option ) {
			$this->UpdateShippingOption($event);
		}
		else {
			$this->UpdateShippingTypes($event);
		}
		$this->RecalculateProcessingFee($event);
		$this->UpdateShippingTotal($event);
		$this->RecalculateGift($event);

		// guess fields from "One Step Checkout" form
		if ( $object->GetDBField('PaymentAccount') ) {
			$order_helper = $this->Application->recallObject('OrderHelper');
			/* @var $order_helper OrderHelper */

			$object->SetDBField(
				'PaymentCardType',
				$order_helper->getCreditCardType($object->GetDBField('PaymentAccount'))
			);
		}
		else {
			$object->SetDBField('PaymentCardType', '');
		}

		if ( !$object->GetDBField('PaymentNameOnCard') ) {
			$object->SetDBField('PaymentNameOnCard', $object->GetDBField('BillingTo'));
		}

		if ( is_object($event->MasterEvent) && $event->MasterEvent->Name == 'OnUpdateAjax' && $this->Application->GetVar('create_account') && $object->Validate() ) {
			$this->createAccountFromOrder($event);
		}
	}

	/**
	 * Creates user account
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function createAccountFromOrder($event)
	{
		$order = $event->getObject();
		/* @var $order OrdersItem */

		$order_helper = $this->Application->recallObject('OrderHelper');
		/* @var $order_helper OrderHelper */

		$user_fields = $order_helper->getUserFields($order);
		$user_fields['Password'] = $order->GetDBField('UserPassword_plain');
		$user_fields['VerifyPassword'] = $order->GetDBField('VerifyUserPassword_plain');

		if ( $order->GetDBField('PortalUserId') == USER_GUEST ) {
			// will also auto-login user when created
			$this->Application->SetVar('u_register', Array (USER_GUEST => $user_fields));
			$this->Application->HandleEvent(new kEvent('u.register:OnCreate'));
		}
		else {
			$user = $this->Application->recallObject('u.current');
			/* @var $user UsersItem */

			$user->SetFieldsFromHash($user_fields);
			if ( !$user->Update() ) {
				$order->SetError('BillingEmail', $user->GetErrorPseudo('Email'));
			}
		}
	}

	/**
	 * Apply any custom changes to list's sql query
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 * @see kDBEventHandler::OnListBuild()
	 */
	protected function SetCustomQuery(kEvent $event)
	{
		parent::SetCustomQuery($event);

		$object = $event->getObject();
		/* @var $object kDBList */

		$types = $event->getEventParam('types');
		if ( $types == 'myorders' || $types == 'myrecentorders' ) {
			$user_id = $this->Application->RecallVar('user_id');
			$object->addFilter('myitems_user1', '%1$s.PortalUserId = ' . $user_id);
			$object->addFilter('myitems_user2', '%1$s.PortalUserId > 0');
			$object->addFilter('Status', '%1$s.Status != 0');
		}
		else if ($event->Special == 'returns') {
//			$object->addFilter('returns_filter',TABLE_PREFIX.'Orders.Status = '.ORDER_STATUS_PROCESSED.' AND (
//				SELECT SUM(ReturnType)
//				FROM '.TABLE_PREFIX.'OrderItems oi
//				WHERE oi.OrderId = '.TABLE_PREFIX.'Orders.OrderId
//			) > 0');
			$object->addFilter('returns_filter', TABLE_PREFIX . 'Orders.Status = ' . ORDER_STATUS_PROCESSED . ' AND ' . TABLE_PREFIX . 'Orders.ReturnTotal > 0');
		}
		else if ( $event->Special == 'user' ) {
			$user_id = $this->Application->GetVar('u_id');
			$object->addFilter('user_filter', '%1$s.PortalUserId = ' . $user_id);
		}
		else {
			$special = $event->Special ? $event->Special : $this->Application->GetVar('order_type');
			if ( $special != 'search' ) {
				// don't filter out orders by special in case of search tab
				$object->addFilter('status_filter', '%1$s.Status=' . $this->getTypeBySpecial($special));
			}

			if ( $event->getEventParam('selected_only') ) {
				$ids = $this->StoreSelectedIDs($event);
				$object->addFilter('selected_filter', '%1$s.OrderId IN (' . implode(',', $ids) . ')');
			}
		}
	}

	function getTypeBySpecial($special)
	{
		$special2type = Array('incomplete'=>0,'pending'=>1,'backorders'=>2,'toship'=>3,'processed'=>4,'denied'=>5,'archived'=>6);
		return $special2type[$special];
	}

	function getSpecialByType($type)
	{
		$type2special = Array(0=>'incomplete',1=>'pending',2=>'backorders',3=>'toship',4=>'processed',5=>'denied',6=>'archived');
		return $type2special[$type];
	}

	function LockTables($event)
	{
		$read = Array();
		$write_lock = '';
		$read_lock = '';
		$write = Array('Orders','OrderItems','Products');
		foreach ($write as $tbl) {
			$write_lock .= TABLE_PREFIX.$tbl.' WRITE,';
		}
		foreach ($read as $tbl) {
			$read_lock .= TABLE_PREFIX.$tbl.' READ,';
		}
		$write_lock = rtrim($write_lock, ',');
		$read_lock = rtrim($read_lock, ',');
		$lock = trim($read_lock.','.$write_lock, ',');
		//$this->Conn->Query('LOCK TABLES '.$lock);
	}

	/**
	 * Checks shopping cart products quantities
	 *
	 * @param kEvent $event
	 * @return bool
	 */
	function CheckQuantites($event)
	{
		if ( $this->OnRecalculateItems($event) ) { // if something has changed in the order
			if ( $this->Application->isAdminUser ) {
				if ( $this->UseTempTables($event) ) {
					$event->redirect = 'in-commerce/orders/orders_edit_items';
				}
			}
			else {
				$event->redirect = $this->Application->GetVar('viewcart_template');
			}

			return false;
		}

		return true;
	}

	function DoPlaceOrder($event)
	{
		$order = $event->getObject();

		$table_prefix = $this->TablePrefix($event);

		$this->LockTables($event);

		if (!$this->CheckQuantites($event)) return false;

		//everything is fine - we could reserve items
		$this->ReserveItems($event);
		$this->SplitOrder($event, $order);
		return true;
	}

	function &queryOrderItems($event, $table_prefix)
	{
		$order = $event->getObject();
		$ord_id = $order->GetId();

		// TABLE_PREFIX and $table_prefix are NOT the same !!!
		$poc_table = $this->Application->getUnitOption('poc', 'TableName');
		$query = '	SELECT
							 BackOrderFlag, '.
							 $table_prefix.'OrderItems.OrderItemId, '.
							 $table_prefix.'OrderItems.Quantity, '.
							 $table_prefix.'OrderItems.QuantityReserved,
							 IF('.TABLE_PREFIX.'Products.InventoryStatus = 2, '.$poc_table.'.QtyInStock, '.TABLE_PREFIX.'Products.QtyInStock) AS QtyInStock, '.
							 TABLE_PREFIX.'Products.QtyInStockMin, '.
							 $table_prefix.'OrderItems.ProductId, '.
							 TABLE_PREFIX.'Products.InventoryStatus,'.
							 $table_prefix.'OrderItems.OptionsSalt AS CombinationCRC
					FROM '.$table_prefix.'OrderItems
					LEFT JOIN '.TABLE_PREFIX.'Products ON '.TABLE_PREFIX.'Products.ProductId = '.$table_prefix.'OrderItems.ProductId
					LEFT JOIN '.$poc_table.' ON ('.$poc_table.'.CombinationCRC = '.$table_prefix.'OrderItems.OptionsSalt) AND ('.$poc_table.'.ProductId = '.$table_prefix.'OrderItems.ProductId)
					WHERE OrderId = '.$ord_id.' AND '.TABLE_PREFIX.'Products.Type = 1
					ORDER BY BackOrderFlag ASC';

		$items = $this->Conn->Query($query);
		return $items;
	}

	function ReserveItems($event)
	{
		$table_prefix = $this->TablePrefix($event);
		$items =& $this->queryOrderItems($event, $table_prefix);

		foreach ($items as $an_item) {
			if (!$an_item['InventoryStatus']) {
				$to_reserve = $an_item['Quantity'] - $an_item['QuantityReserved'];
			}
			else {
				if ($an_item['BackOrderFlag'] > 0) { // we don't need to reserve if it's backordered item
					$to_reserve = 0;
				}
				else {
					$to_reserve = min($an_item['Quantity']-$an_item['QuantityReserved'], $an_item['QtyInStock']-$an_item['QtyInStockMin']); //it should be equal, but just in case
				}

				$to_backorder = $an_item['BackOrderFlag'] > 0 ? $an_item['Quantity']-$an_item['QuantityReserved'] : 0;
			}

			if ($to_backorder < 0) $to_backorder = 0; //just in case
			$query = '	UPDATE '.$table_prefix.'OrderItems
						SET QuantityReserved = IF(QuantityReserved IS NULL, '.$to_reserve.', QuantityReserved + '.$to_reserve.')
						WHERE OrderItemId = '.$an_item['OrderItemId'];
			$this->Conn->Query($query);

			if (!$an_item['InventoryStatus']) continue;

			$update_clause = '	QtyInStock = QtyInStock - '.$to_reserve.',
							  	QtyReserved = QtyReserved + '.$to_reserve.',
								QtyBackOrdered = QtyBackOrdered + '.$to_backorder;

			if ($an_item['InventoryStatus'] == 1) {
				// inventory by product, then update it's quantities
				$query = '	UPDATE '.TABLE_PREFIX.'Products
							SET '.$update_clause.'
								WHERE ProductId = '.$an_item['ProductId'];
			}
			else {
				// inventory = 2 -> by product option combinations
				$poc_idfield = $this->Application->getUnitOption('poc', 'IDField');
				$poc_table = $this->Application->getUnitOption('poc', 'TableName');
				$query = '	UPDATE '.$poc_table.'
							SET '.$update_clause.'
							WHERE (ProductId = '.$an_item['ProductId'].') AND (CombinationCRC = '.$an_item['CombinationCRC'].')';
			}
			$this->Conn->Query($query);
		}
	}

	function FreeItems($event)
	{
		$table_prefix = $this->TablePrefix($event);
		$items =& $this->queryOrderItems($event, $table_prefix);

		foreach ($items as $an_item) {
				$to_free = $an_item['QuantityReserved'];

				if ($an_item['InventoryStatus']) {
				if ($an_item['BackOrderFlag'] > 0) { // we don't need to free if it's backordered item
					$to_free = 0;
				}

				// what's not reserved goes to backorder in stock for orderitems marked with BackOrderFlag
				$to_backorder_free = $an_item['BackOrderFlag'] > 0 ? $an_item['Quantity'] - $an_item['QuantityReserved'] : 0;
				if ($to_backorder_free < 0) $to_backorder_free = 0; //just in case

				$update_clause = '	QtyInStock = QtyInStock + '.$to_free.',
								  	QtyReserved = QtyReserved - '.$to_free.',
							  		QtyBackOrdered = QtyBackOrdered - '.$to_backorder_free;

				if ($an_item['InventoryStatus'] == 1) {
					// inventory by product
					$query = '	UPDATE '.TABLE_PREFIX.'Products
								SET '.$update_clause.'
									WHERE ProductId = '.$an_item['ProductId'];
				}
				else {
					// inventory by option combinations
					$poc_idfield = $this->Application->getUnitOption('poc', 'IDField');
					$poc_table = $this->Application->getUnitOption('poc', 'TableName');
					$query = '	UPDATE '.$poc_table.'
								SET '.$update_clause.'
								WHERE (ProductId = '.$an_item['ProductId'].') AND (CombinationCRC = '.$an_item['CombinationCRC'].')';
				}

					$this->Conn->Query($query);
				}

			$query = '	UPDATE '.$table_prefix.'OrderItems
						SET QuantityReserved = IF(QuantityReserved IS NULL, 0, QuantityReserved - '.$to_free.')
									WHERE OrderItemId = '.$an_item['OrderItemId'];
				$this->Conn->Query($query);
		}
	}

	/**
	 * Enter description here...
	 *
	 * @param kEvent $event
	 * @param OrdersItem $object
	 */
	function SplitOrder($event, &$object)
	{
		$affiliate_event = new kEvent('affil:OnOrderApprove');
		$affiliate_event->setEventParam('Order_PrefixSpecial', $object->getPrefixSpecial() );
		$this->Application->HandleEvent($affiliate_event);

		$table_prefix = $this->TablePrefix($event);
		$order =& $object;
		$ord_id = $order->GetId();

		$shipping_option = $order->GetDBField('ShippingOption');
		$backorder_select = $shipping_option == 0 ? '0' : '%s.BackOrderFlag';


		// setting PackageNum to 0 for Non-tangible items, for tangibles first package num is always 1
		$query = '	SELECT oi.OrderItemId
					FROM ' . $table_prefix . 'OrderItems oi
					LEFT JOIN ' . TABLE_PREFIX . 'Products p ON p.ProductId = oi.ProductId
					WHERE p.Type > 1 AND oi.OrderId = ' . $ord_id;
		$non_tangibles = $this->Conn->GetCol($query);

		if ($non_tangibles) {
			$query = '	UPDATE ' . $table_prefix . 'OrderItems
						SET PackageNum = 0
						WHERE OrderItemId IN (' . implode(',', $non_tangibles) . ')';
			$this->Conn->Query($query);
		}

		// grouping_data:
		// 0 => Product Type
		// 1 => if NOT tangibale and NOT downloadable - OrderItemId,
		//			2 => ProductId
		// 3 => Shipping PackageNum
		$query = 'SELECT
					' . sprintf($backorder_select, $table_prefix . 'OrderItems') . ' AS BackOrderFlagCalc,
					PackageNum,
					ProductName,
					ShippingTypeId,
					CONCAT('.TABLE_PREFIX.'Products.Type,
						"_",
						IF ('.TABLE_PREFIX.'Products.Type NOT IN ('.PRODUCT_TYPE_DOWNLOADABLE.','.PRODUCT_TYPE_TANGIBLE.'),
							CONCAT(OrderItemId, "_", '.TABLE_PREFIX.'Products.ProductId),
							""),
						"_",
						PackageNum
						) AS Grouping,
					SUM(Quantity) AS TotalItems,
					SUM('.$table_prefix.'OrderItems.Weight*Quantity) AS TotalWeight,
					SUM(Price * Quantity) AS TotalAmount,
					SUM(QuantityReserved) AS TotalReserved,
					'.TABLE_PREFIX.'Products.Type AS ProductType
				FROM '.$table_prefix.'OrderItems
				LEFT JOIN '.TABLE_PREFIX.'Products
					ON '.TABLE_PREFIX.'Products.ProductId = '.$table_prefix.'OrderItems.ProductId
				WHERE OrderId = '.$ord_id.'
				GROUP BY BackOrderFlagCalc, Grouping
				ORDER BY BackOrderFlagCalc ASC, PackageNum ASC, ProductType ASC';

		$sub_orders = $this->Conn->Query($query);

		$processed_sub_orders = Array();

		// in case of recurring billing this will not be 0 as usual
		//$first_sub_number = ($event->Special == 'recurring') ? $object->getNextSubNumber() - 1 : 0;
		$first_sub_number = $object->GetDBField('SubNumber');

		$next_sub_number = $first_sub_number;
		$group = 1;

		$order_has_gift = $order->GetDBField('GiftCertificateDiscount') > 0 ? 1 : 0;

		$skip_types = Array (PRODUCT_TYPE_TANGIBLE, PRODUCT_TYPE_DOWNLOADABLE);
		foreach ($sub_orders as $sub_order_data) {
			$sub_order = $this->Application->recallObject('ord.-sub'.$next_sub_number, 'ord');
			/* @var $sub_order OrdersItem */

			if ( $this->UseTempTables($event) && $next_sub_number == 0 ) {
				$sub_order =& $order;
			}
			else {
				foreach ( $order->GetFieldValues() as $field => $value ) {
					$sub_order->SetOriginalField($field, $value);
				}
			}

			$sub_order->SetDBFieldsFromHash($order->GetFieldValues());
			$sub_order->SetDBField('SubNumber', $next_sub_number);
			$sub_order->SetDBField('SubTotal', $sub_order_data['TotalAmount']);

			$grouping_data = explode('_', $sub_order_data['Grouping']);
			$named_grouping_data['Type'] = $grouping_data[0];

			if (!in_array($named_grouping_data['Type'], $skip_types)) {
				$named_grouping_data['OrderItemId'] = $grouping_data[1];
				$named_grouping_data['ProductId'] = $grouping_data[2];
				$named_grouping_data['PackageNum'] = $grouping_data[3];
			}
			else {
				$named_grouping_data['PackageNum'] = $grouping_data[2];
			}

			if ($named_grouping_data['Type'] == PRODUCT_TYPE_TANGIBLE) {
				$sub_order->SetDBField('ShippingCost', getArrayValue( unserialize($order->GetDBField('ShippingInfo')), $sub_order_data['PackageNum'], 'TotalCost') );
				$sub_order->SetDBField('InsuranceFee', getArrayValue( unserialize($order->GetDBField('ShippingInfo')), $sub_order_data['PackageNum'], 'InsuranceFee') );
				$sub_order->SetDBField('ShippingInfo', serialize(Array(1 => getArrayValue( unserialize($order->GetDBField('ShippingInfo')), $sub_order_data['PackageNum']))));
			}
			else {
				$sub_order->SetDBField('ShippingCost', 0);
				$sub_order->SetDBField('InsuranceFee', 0);
				$sub_order->SetDBField('ShippingInfo', ''); //otherwise orders w/o shipping wills still have shipping info!
			}

			$amount_percent = $sub_order->getTotalAmount() * 100 / $order->getTotalAmount();
			// proportional affiliate commission splitting
			if ($order->GetDBField('AffiliateCommission') > 0) {
				$sub_order->SetDBField('AffiliateCommission', $order->GetDBField('AffiliateCommission') * $amount_percent / 100 );
			}

			$amount_percent = ($sub_order->GetDBField('SubTotal') + $sub_order->GetDBField('ShippingCost')) * 100 / ($order->GetDBField('SubTotal') + $order->GetDBField('ShippingCost'));
			if ($order->GetDBField('ProcessingFee') > 0) {
				$sub_order->SetDBField('ProcessingFee', round($order->GetDBField('ProcessingFee') * $amount_percent / 100, 2));
			}

			$sub_order->RecalculateTax();

			$original_amount = $sub_order->GetDBField('SubTotal') + $sub_order->GetDBField('ShippingCost') + $sub_order->GetDBField('VAT') + $sub_order->GetDBField('ProcessingFee') + $sub_order->GetDBField('InsuranceFee') - $sub_order->GetDBField('GiftCertificateDiscount');
			$sub_order->SetDBField('OriginalAmount', $original_amount);

			if ($named_grouping_data['Type'] == 1 && ($sub_order_data['BackOrderFlagCalc'] > 0
					||
					($sub_order_data['TotalItems'] != $sub_order_data['TotalReserved'])) ) {
				$sub_order->SetDBField('Status', ORDER_STATUS_BACKORDERS);

				if ($event->Special != 'recurring') { // just in case if admin uses tangible backordered products in recurring orders
					$this->Application->emailUser('BACKORDER.ADD', null, $this->OrderEmailParams($sub_order));
		    		$this->Application->emailAdmin('BACKORDER.ADD');
				}
			}
			else {
				switch ($named_grouping_data['Type']) {
					case PRODUCT_TYPE_DOWNLOADABLE:
						$sql = 'SELECT oi.*
								FROM '.TABLE_PREFIX.'OrderItems oi
								LEFT JOIN '.TABLE_PREFIX.'Products p ON p.ProductId = oi.ProductId
								WHERE (OrderId = %s) AND (p.Type = '.PRODUCT_TYPE_DOWNLOADABLE.')';
						$downl_products = $this->Conn->Query( sprintf($sql, $ord_id) );
						$product_ids = Array();
						foreach ($downl_products as $downl_product) {
							$this->raiseProductEvent('Approve', $downl_product['ProductId'], $downl_product, $next_sub_number);
							$product_ids[] = $downl_product['ProductId'];
						}
						break;

					case PRODUCT_TYPE_TANGIBLE:
						$sql = 'SELECT ' . sprintf($backorder_select, 'oi') . ' AS BackOrderFlagCalc, oi.*
								FROM ' . TABLE_PREFIX . 'OrderItems oi
								LEFT JOIN ' . TABLE_PREFIX . 'Products p ON p.ProductId = oi.ProductId
								WHERE (OrderId = %s) AND (p.Type = ' . PRODUCT_TYPE_TANGIBLE . ')
								HAVING BackOrderFlagCalc = 0';

							$products = $this->Conn->Query( sprintf($sql, $ord_id) );
							foreach ($products as $product) {
								$this->raiseProductEvent('Approve', $product['ProductId'], $product, $next_sub_number);
							}
						break;

					default:
						$order_item_fields = $this->Conn->GetRow('SELECT * FROM '.TABLE_PREFIX.'OrderItems WHERE OrderItemId = '.$named_grouping_data['OrderItemId']);
						$this->raiseProductEvent('Approve', $named_grouping_data['ProductId'], $order_item_fields, $next_sub_number);
						break;
				}

				$sub_order->SetDBField('Status', $named_grouping_data['Type'] == PRODUCT_TYPE_TANGIBLE ? ORDER_STATUS_TOSHIP : ORDER_STATUS_PROCESSED);
			}

			if ($next_sub_number == $first_sub_number) {
				$sub_order->SetId($order->GetId());
				$sub_order->Update();
			}
			else {
				$sub_order->Create();
			}

			switch ($named_grouping_data['Type']) {
				case PRODUCT_TYPE_TANGIBLE:
					$query = '	UPDATE ' . $table_prefix . 'OrderItems
								SET OrderId = %s, PackageNum = 1
								WHERE OrderId = %s AND PackageNum = %s';
					$query = sprintf($query, $sub_order->GetID(), $ord_id, $sub_order_data['PackageNum']);
					break;

				case PRODUCT_TYPE_DOWNLOADABLE:
					$query = '	UPDATE ' . $table_prefix . 'OrderItems
								SET OrderId = %s, PackageNum = 1
								WHERE OrderId = %s AND ProductId IN (%s)';
					$query = sprintf($query, $sub_order->GetID(), $ord_id, implode(',', $product_ids));
					break;

				default:
					$query = '	UPDATE ' . $table_prefix . 'OrderItems
								SET OrderId = %s, PackageNum = 1
								WHERE OrderId = %s AND OrderItemId = %s';
					$query = sprintf($query, $sub_order->GetID(), $ord_id, $named_grouping_data['OrderItemId']);
					break;
			}

			$this->Conn->Query($query);

			if ($order_has_gift) {
				// gift certificate can be applied only after items are assigned to suborder
				$sub_order->RecalculateGift($event);
				$original_amount = $sub_order->GetDBField('SubTotal') + $sub_order->GetDBField('ShippingCost') + $sub_order->GetDBField('VAT') + $sub_order->GetDBField('ProcessingFee') + $sub_order->GetDBField('InsuranceFee') - $sub_order->GetDBField('GiftCertificateDiscount');
				$sub_order->SetDBField('OriginalAmount', $original_amount);
				$sub_order->Update();
			}

			$processed_sub_orders[] = $sub_order->GetID();

			$next_sub_number = $sub_order->getNextSubNumber();
			$group++;
		}

		foreach ($processed_sub_orders as $sub_id) {
			// update DiscountTotal field
			$sql = 'SELECT SUM(ROUND(FlatPrice-Price,2)*Quantity) FROM '.$table_prefix.'OrderItems WHERE OrderId = '.$sub_id;
			$discount_total = $this->Conn->GetOne($sql);

			$sql = 'UPDATE '.$sub_order->TableName.'
					SET DiscountTotal = '.$this->Conn->qstr($discount_total).'
					WHERE OrderId = '.$sub_id;
			$this->Conn->Query($sql);
		}
	}

	/**
	 * Call products linked event when spefcfic action is made to product in order
	 *
	 * @param string $event_type type of event to get from product ProcessingData = {Approve,Deny,CompleteOrder}
	 * @param int $product_id ID of product to gather processing data from
	 * @param Array $order_item_fields OrderItems table record fields (with needed product & order in it)
	 */
	function raiseProductEvent($event_type, $product_id, $order_item_fields, $next_sub_number=null)
	{
		$sql = 'SELECT ProcessingData
				FROM '.TABLE_PREFIX.'Products
				WHERE ProductId = '.$product_id;
		$processing_data = $this->Conn->GetOne($sql);
		if ($processing_data) {
			$processing_data = unserialize($processing_data);
			$event_key = getArrayValue($processing_data, $event_type.'Event');
			// if requested type of event is defined for product, only then process it
			if ($event_key) {
				$event = new kEvent($event_key);
				$event->setEventParam('field_values', $order_item_fields);
				$event->setEventParam('next_sub_number', $next_sub_number);
				$this->Application->HandleEvent($event);
			}
		}
	}

	function OptionsSalt($options, $comb_only=false)
	{
		$helper = $this->Application->recallObject('kProductOptionsHelper');
		return $helper->OptionsSalt($options, $comb_only);
	}

	/**
	 * Enter description here...
	 *
	 * @param kEvent $event
	 * @param int $item_id
	 */
	function AddItemToOrder($event, $item_id, $qty = null, $package_num = null)
	{
		if (!isset($qty)) {
			$qty = 1;
		}

		// Loading product to add
		$product = $this->Application->recallObject('p.toadd', null, Array('skip_autoload' => true));
		/* @var $product kDBItem */

		$product->Load($item_id);

		$object = $this->Application->recallObject('orditems.-item', null, Array('skip_autoload' => true));
		/* @var $object kDBItem */

		$order = $this->Application->recallObject('ord');
		/* @var $order kDBItem */

		if (!$order->isLoaded() && !$this->Application->isAdmin) {
			// no order was created before -> create one now
			if ($this->_createNewCart($event)) {
				$this->LoadItem($event);
			}
		}

		if (!$order->isLoaded()) {
			// was unable to create new order
			return false;
		}

		$item_data = $event->getEventParam('ItemData');
		$item_data = $item_data ? unserialize($item_data) : Array ();
		$options = getArrayValue($item_data, 'Options');

		if ( !$this->CheckOptions($event, $options, $item_id, $qty, $product->GetDBField('OptionsSelectionMode')) ) {
			return;
		}

		$manager = $this->Application->recallObject('OrderManager');
		/* @var $manager OrderManager */

		$manager->setOrder($order);
		$manager->addProduct($product, $event->getEventParam('ItemData'), $qty, $package_num);

		$this->Application->HandleEvent(new kEvent('ord:OnRecalculateItems'));
	}

	/**
	 * Enter description here...
	 *
	 * @param kEvent $event
	 */
	function UpdateShippingTotal($event)
	{
		if ( $this->Application->GetVar('ebay_notification') == 1 ) {
			// TODO: get rid of this "if"
			return;
		}

		$object = $event->getObject();
		/* @var $object OrdersItem */

		$shipping_total = $insurance_fee = 0;
		$shipping_info = $object->GetDBField('ShippingInfo') ? unserialize($object->GetDBField('ShippingInfo')) : false;

		if ( is_array($shipping_info) ) {
			foreach ($shipping_info as $a_shipping) {
//				$id_elements = explode('_', $a_shipping['ShippingTypeId']);
				$shipping_total += $a_shipping['TotalCost'];
				$insurance_fee += $a_shipping['InsuranceFee'];
			}
		}

		$object->SetDBField('ShippingCost', $shipping_total);
		$object->SetDBField('InsuranceFee', $insurance_fee);
		// no need to update, it will be called in calling method

		$this->RecalculateTax($event);
	}

	/**
	 * Recompile shopping cart, splitting or grouping orders and backorders depending on total quantities.
	 * First it counts total qty for each ProductId, and then creates order for available items
	 * and backorder for others. It also updates the sub-total for the order
	 *
	 * @param kEvent $event
	 * @return bool Returns true if items splitting/grouping were changed
	 */
	function OnRecalculateItems($event)
	{
		if (is_object($event->MasterEvent) && ($event->MasterEvent->status != kEvent::erSUCCESS)) {
			// e.g. master order update failed, don't recalculate order products
			return ;
		}

		$order = $event->getObject();
		/* @var $order OrdersItem */

		if ( !$order->isLoaded() ) {
			$this->LoadItem($event); // try to load
		}

		$ord_id = (int)$order->GetID();

		if ( !$order->isLoaded() ) return; //order has not been created yet

		if( $order->GetDBField('Status') != ORDER_STATUS_INCOMPLETE )
		{
			return;
		}

		$manager = $this->Application->recallObject('OrderManager');
		/* @var $manager OrderManager */

		$manager->setOrder($order);
		$result = $manager->calculate();

		if ( $order->GetDBField('CouponId') && $order->GetDBField('CouponDiscount') == 0 ) {
			$this->RemoveCoupon($order);
			$order->setCheckoutError(OrderCheckoutErrorType::COUPON, OrderCheckoutError::COUPON_REMOVED_AUTOMATICALLY);
		}

		if ( $result ) {
			$this->UpdateShippingOption($event);
		}

		$this->UpdateShippingTotal($event);

		$this->RecalculateProcessingFee($event);
		$this->RecalculateTax($event);
		$this->RecalculateGift($event);

		if ( $event->Name != 'OnAfterItemUpdate' ) {
			$order->Update();
		}

		$event->setEventParam('RecalculateChangedCart', $result);

		if ( is_object($event->MasterEvent) ) {
			$event->MasterEvent->setEventParam('RecalculateChangedCart', $result);
		}

		/*if ( $result && !getArrayValue($event->redirect_params, 'checkout_error') ) {
			$event->SetRedirectParam('checkout_error', OrderCheckoutError::STATE_CHANGED);
		}*/

		if ( $result && is_object($event->MasterEvent) && $event->MasterEvent->Name == 'OnUserLogin' ) {
			$shop_cart_template = $this->Application->GetVar('shop_cart_template');

			if ( $shop_cart_template && is_object($event->MasterEvent->MasterEvent) ) {
//				$event->MasterEvent->MasterEvent->SetRedirectParam('checkout_error', OrderCheckoutError::CHANGED_AFTER_LOGIN);
				$event->MasterEvent->MasterEvent->redirect = $shop_cart_template;
			}
		}

		return $result;
	}

/*	function GetShippingCost($user_country_id, $user_state_id, $user_zip, $weight, $items, $amount, $shipping_type)
	{
		$this->Application->recallObject('ShippingQuoteEngine');
		$shipping_h = $this->Application->recallObject('CustomShippingQuoteEngine');
		$query = $shipping_h->QueryShippingCost($user_country_id, $user_state_id, $user_zip, $weight, $items, $amount, $shipping_type);
		$cost = $this->Conn->GetRow($query);
		return $cost['TotalCost'];
	}*/

	/**
	 * Return product pricing id for given product, if not passed - return primary pricing ID
	 *
	 * @param int $product_id ProductId
	 * @return float
	 */
	function GetPricingId($product_id, $item_data)	{

		if (!is_array($item_data)) {
			$item_data = unserialize($item_data);
		}
		$price_id = getArrayValue($item_data, 'PricingId');
		if (!$price_id) {
		$price_id = $this->Application->GetVar('pr_id');
		}
		if (!$price_id){
			$price_id = $this->Conn->GetOne('SELECT PriceId FROM '.TABLE_PREFIX.'ProductsPricing WHERE ProductId='.$product_id.' AND IsPrimary=1');
		}
		return $price_id;
	}

	function UpdateShippingOption($event)
	{
		$object = $event->getObject();
		$shipping_option = $object->GetDBField('ShippingOption');

		if($shipping_option == '') return;

		$table_prefix = $this->TablePrefix($event);

		if ($shipping_option == 1 || $shipping_option == 0) { // backorder separately
			$query = 'UPDATE '.$table_prefix.'OrderItems SET BackOrderFlag = 1 WHERE OrderId = '.$object->GetId().' AND BackOrderFlag > 1';
			$this->Conn->Query($query);
		}
		if ($shipping_option == 2) {
			$query = 'SELECT * FROM '.$table_prefix.'OrderItems WHERE OrderId = '.$object->GetId().' AND BackOrderFlag >= 1 ORDER By ProductName asc';
			$items = $this->Conn->Query($query);
			$backorder_flag = 2;
			foreach ($items as $an_item) {
				$query = 'UPDATE '.$table_prefix.'OrderItems SET BackOrderFlag = '.$backorder_flag.' WHERE OrderItemId = '.$an_item['OrderItemId'];
				$this->Conn->Query($query);
				$backorder_flag++;
			}
		}
	}

	/**
	 * Updates shipping types
	 *
	 * @param kEvent $event
	 * @return bool
	 */
	function UpdateShippingTypes($event)
	{
		$object = $event->getObject();
		/* @var $object OrdersItem */

		$ord_id = $object->GetID();

		$order_info = $this->Application->GetVar('ord');
		$shipping_ids = getArrayValue($order_info, $ord_id, 'ShippingTypeId');

		if (!$shipping_ids) {
			return;
		}

		$ret = true;
		$shipping_types = Array();
		$last_shippings = unserialize( $this->Application->RecallVar('LastShippings') );

		$template = $this->Application->GetVar('t');
		$shipping_templates = Array ('in-commerce/checkout/shipping', 'in-commerce/orders/orders_edit_shipping');

		$quote_engine_collector = $this->Application->recallObject('ShippingQuoteCollector');
		/* @var $quote_engine_collector ShippingQuoteCollector */

		foreach ($shipping_ids as $package => $id) {
			// try to validate
			$shipping_types[$package] = $last_shippings[$package][$id];
			$sqe_class_name = $quote_engine_collector->GetClassByType($shipping_types, $package);

			if (($object->GetDBField('ShippingType') == 0) && ($sqe_class_name !== false) && in_array($template, $shipping_templates)) {
				$shipping_quote_engine = $this->Application->recallObject($sqe_class_name);
				/* @var $shipping_quote_engine ShippingQuoteEngine */

				// USPS related part
				// TODO: remove USPS condition from here
				// set first of found shippings just to check if any errors are returned
				$current_usps_shipping_types = unserialize($this->Application->RecallVar('current_usps_shipping_types'));
				$object->SetDBField('ShippingInfo', serialize( Array($package => $current_usps_shipping_types[$id])) );

				$sqe_data = $shipping_quote_engine->MakeOrder($object, true);

				if ( $sqe_data ) {
					if ( !isset($sqe_data['error_number']) ) {
						// update only international shipping
						if ( $object->GetDBField('ShippingCountry') != 'USA') {
							$shipping_types[$package]['TotalCost'] = $sqe_data['Postage'];
						}
					}
					else {
						$ret = false;
						$this->Application->StoreVar('sqe_error', $sqe_data['error_description']);
					}
				}

				$object->SetDBField('ShippingInfo', '');
			}
		}

		$object->SetDBField('ShippingInfo', serialize($shipping_types));

		return $ret;
	}

	/*function shipOrder(&$order_items)
	{
		$product_object = $this->Application->recallObject('p', null, Array('skip_autoload' => true));
		$order_item = $this->Application->recallObject('orditems.-item');

		while( !$order_items->EOL() )
		{
			$rec = $order_items->getCurrentRecord();

			$order_item->SetDBFieldsFromHash($rec);
			$order_item->SetId($rec['OrderItemId']);
			$order_item->SetDBField('QuantityReserved', 0);
			$order_item->Update();

			$order_items->GoNext();
		}
		return true;
	}*/

	function RecalculateTax($event)
	{
		$object = $event->getObject();
		/* @var $object OrdersItem */

		if ($object->GetDBField('Status') > ORDER_STATUS_PENDING) {
			return;
		}

		$object->RecalculateTax();
	}

	function RecalculateProcessingFee($event)
	{
		$object = $event->getObject();

		// Do not reset processing fee while orders are being split (see SplitOrder)
		if (preg_match("/^-sub/", $object->Special)) return;
		if ($object->GetDBField('Status') > ORDER_STATUS_PENDING) return; //no changes for orders other than incomple or pending

		$pt = $object->GetDBField('PaymentType');
		$processing_fee = $this->Conn->GetOne('SELECT ProcessingFee FROM '.$this->Application->getUnitOption('pt', 'TableName').' WHERE PaymentTypeId = '.$pt);
		$object->SetDBField( 'ProcessingFee', $processing_fee );
		$this->UpdateTotals($event);
	}

	function UpdateTotals($event)
	{
		$object = $event->getObject();
		/* @var $object OrdersItem */

		$object->UpdateTotals();
	}

	/*function CalculateDiscount($event)
	{
		$object = $event->getObject();

		$coupon = $this->Application->recallObject('coup', null, Array('skip_autoload' => true));
		if(!$coupon->Load( $object->GetDBField('CouponId'), 'CouponId' ))
		{
			return false;
		}

		$sql = 'SELECT Price * Quantity AS Amount, ProductId FROM '.$this->Application->getUnitOption('orditems', 'TableName').'
				WHERE OrderId = '.$object->GetDBField('OrderId');
		$orditems = $this->Conn->GetCol($sql, 'ProductId');

		$sql = 'SELECT coupi.ItemType, p.ProductId FROM '.$this->Application->getUnitOption('coupi', 'TableName').' coupi
				LEFT JOIN '.$this->Application->getUnitOption('p', 'TableName').' p
				ON coupi.ItemResourceId = p.ResourceId
				WHERE CouponId = '.$object->GetDBField('CouponId');
		$discounts = $this->Conn->GetCol($sql, 'ProductId');

		$discount_amount = 0;

		foreach($orditems as $product_id => $amount)
		{
			if(isset($discounts[$product_id]) || array_search('0', $discounts, true) !== false)
			{
				switch($coupon->GetDBField('Type'))
				{
					case 1:
						$discount_amount += $coupon->GetDBField('Amount') < $amount ? $coupon->GetDBField('Amount') : $amount;
					break;
					case 2:
						$discount_amount += $amount * $coupon->GetDBField('Amount') / 100;
					break;
					default:
				}
				break;
			}
		}

		$object->SetDBField('CouponDiscount', $discount_amount);
		return $discount_amount;
	}*/

	/**
	 * Jumps to selected order in order's list from search tab
	 *
	 * @param kEvent $event
	 */
	function OnGoToOrder($event)
	{
		$id = current($this->StoreSelectedIDs($event));

		$id_field = $this->Application->getUnitOption($event->Prefix,'IDField');
		$table = $this->Application->getUnitOption($event->Prefix,'TableName');

		$sql = 'SELECT Status FROM %s WHERE %s = %s';

		$order_status = $this->Conn->GetOne( sprintf($sql, $table, $id_field, $id) );

		$prefix_special = $event->Prefix.'.'.$this->getSpecialByType($order_status);

		$orders_list = $this->Application->recallObject($prefix_special, $event->Prefix.'_List', Array('per_page'=>-1) );
		/* @var $orders_list kDBList */

		$orders_list->Query();

		foreach ($orders_list->Records as $row_num => $record) {
			if ( $record[$id_field] == $id ) {
				break;
			}
		}

		$per_page = $this->getPerPage( new kEvent($prefix_special.':OnDummy') );
		$page = ceil( ($row_num+1) / $per_page );

		$this->Application->StoreVar($prefix_special.'_Page', $page);
		$event->redirect = 'in-commerce/orders/orders_'.$this->getSpecialByType($order_status).'_list';
	}

	/**
	 * Reset's any selected order state to pending
	 *
	 * @param kEvent $event
	 */
	function OnResetToPending($event)
	{
		$object = $event->getObject( Array('skip_autoload' => true) );
		/* @var $object kDBItem */

		$items_info = $this->Application->GetVar($event->getPrefixSpecial(true));

		if ( $items_info ) {
			foreach ($items_info as $id => $field_values) {
				$object->Load($id);
				$object->SetDBField('Status', ORDER_STATUS_PENDING);

				if ( $object->Update() ) {
					$event->status = kEvent::erSUCCESS;
				}
				else {
					$event->status = kEvent::erFAIL;
					$event->redirect = false;
					break;
				}
			}
		}
	}

	/**
	 * Creates list from items selected in grid
	 *
	 * @param kEvent $event
	 */
	function OnLoadSelected($event)
	{
		$event->setPseudoClass('_List');

		/** @var kDBList $object */
		$object = $event->getObject(array('selected_only' => true, 'per_page' => -1));

		$event->redirect = false;
	}

	/**
	 * Return orders list, that will expire in time specified
	 *
	 * @param int $pre_expiration timestamp
	 * @return Array
	 */
	function getRecurringOrders($pre_expiration)
	{
		$ord_table = $this->Application->getUnitOption('ord', 'TableName');
		$ord_idfield = $this->Application->getUnitOption('ord', 'IDField');

		$processing_allowed = Array(ORDER_STATUS_PROCESSED, ORDER_STATUS_ARCHIVED);
		$sql = 'SELECT '.$ord_idfield.', PortalUserId, GroupId, NextCharge
				FROM '.$ord_table.'
				WHERE (IsRecurringBilling = 1) AND (NextCharge < '.$pre_expiration.') AND Status IN ('.implode(',', $processing_allowed).')';
		return $this->Conn->Query($sql, $ord_idfield);
	}

	/**
	 * [SCHEDULED TASK] Checks what orders should expire and renew automatically (if such flag set)
	 *
	 * @param kEvent $event
	 */
	function OnCheckRecurringOrders($event)
	{
		$skip_clause = Array();
		$ord_table = $this->Application->getUnitOption($event->Prefix, 'TableName');
		$ord_idfield = $this->Application->getUnitOption($event->Prefix, 'IDField');

		$pre_expiration = adodb_mktime() + $this->Application->ConfigValue('Comm_RecurringChargeInverval') * 3600 * 24;
		$to_charge = $this->getRecurringOrders($pre_expiration);
		if ($to_charge) {
			$order_ids = Array();
			foreach ($to_charge as $order_id => $record) {
				// skip virtual users (e.g. root, guest, etc.) & invalid subscriptions (with no group specified, no next charge, but Recurring flag set)
				if (!$record['PortalUserId'] || !$record['GroupId'] || !$record['NextCharge']) continue;

				$order_ids[] = $order_id;
				// prevent duplicate user+group pairs
				$skip_clause[ 'PortalUserId = '.$record['PortalUserId'].' AND GroupId = '.$record['GroupId'] ] = $order_id;
			}

			// process only valid orders
			$temp_handler = $this->Application->recallObject($event->Prefix.'_TempHandler', 'kTempTablesHandler');
			$cloned_order_ids = $temp_handler->CloneItems($event->Prefix, 'recurring', $order_ids);
			$order =&  $this->Application->recallObject($event->Prefix.'.recurring', null, Array('skip_autoload' => true));
			foreach ($cloned_order_ids as $order_id) {
				$order->Load($order_id);
				$this->Application->HandleEvent($complete_event, $event->Prefix.'.recurring:OnCompleteOrder' );

				if ($complete_event->status == kEvent::erSUCCESS) {
					//send recurring ok email
					$this->Application->emailUser('ORDER.RECURRING.PROCESSED', null, $this->OrderEmailParams($order));
					$this->Application->emailAdmin('ORDER.RECURRING.PROCESSED');
				}
				else {
					//send Recurring failed event
					$order->SetDBField('Status', ORDER_STATUS_DENIED);
					$order->Update();
					$this->Application->emailUser('ORDER.RECURRING.DENIED', null, $this->OrderEmailParams($order));
					$this->Application->emailAdmin('ORDER.RECURRING.DENIED');
				}
			}

			// remove recurring flag from all orders found, not to select them next time script runs
			$sql = 'UPDATE '.$ord_table.'
					SET IsRecurringBilling = 0
					WHERE '.$ord_idfield.' IN ('.implode(',', array_keys($to_charge)).')';
			$this->Conn->Query($sql);
		}

		if ( !is_object($event->MasterEvent) ) {
			// not called as hook
			return ;
		}

		$pre_expiration = adodb_mktime() + $this->Application->ConfigValue('User_MembershipExpirationReminder') * 3600 * 24;
		$to_charge = $this->getRecurringOrders($pre_expiration);

		foreach ($to_charge as $order_id => $record) {
			// skip virtual users (e.g. root, guest, etc.) & invalid subscriptions (with no group specified, no next charge, but Recurring flag set)
			if (!$record['PortalUserId'] || !$record['GroupId'] || !$record['NextCharge']) continue;

			// prevent duplicate user+group pairs
			$skip_clause[ 'PortalUserId = '.$record['PortalUserId'].' AND GroupId = '.$record['GroupId'] ] = $order_id;
		}
		$skip_clause = array_flip($skip_clause);

		$event->MasterEvent->setEventParam('skip_clause', $skip_clause);
	}


	function OnGeneratePDF($event)
	{
		$this->OnLoadSelected($event);

		$this->Application->InitParser();
		$o = $this->Application->ParseBlock(array('name'=>'in-commerce/orders/orders_pdf'));

		$file_helper = $this->Application->recallObject('FileHelper');
		/* @var $file_helper FileHelper */

		$file_helper->CheckFolder(EXPORT_PATH);

		$htmlFile = EXPORT_PATH . '/tmp.html';
		$fh = fopen($htmlFile, 'w');
		fwrite($fh, $o);
		fclose($fh);
//		return;


//		require_once (FULL_PATH.'html2pdf/PDFEncryptor.php');

		// Full path to the file to be converted
//		$htmlFile = dirname(__FILE__) . '/test.html';

		// The default domain for images that use a relative path
		// (you'll need to change the paths in the test.html page
		// to an image on your server)
		$defaultDomain = DOMAIN;
		// Full path to the PDF we are creating
		$pdfFile = EXPORT_PATH . '/tmp.pdf';
		// Remove old one, just to make sure we are making it afresh
		@unlink($pdfFile);


		$pdf_helper = $this->Application->recallObject('kPDFHelper');
		$pdf_helper->FileToFile($htmlFile, $pdfFile);
		return ;

		// DOM PDF VERSION
		/*require_once(FULL_PATH.'/dompdf/dompdf_config.inc.php');
		$dompdf = new DOMPDF();
		$dompdf->load_html_file($htmlFile);
		if ( isset($base_path) ) {
		  $dompdf->set_base_path($base_path);
		}
		$dompdf->set_paper($paper, $orientation);
		$dompdf->render();
		file_put_contents($pdfFile, $dompdf->output());
		return ;*/

		// Instnatiate the class with our variables
		require_once (FULL_PATH.'/html2pdf/HTML_ToPDF.php');
		$pdf = new HTML_ToPDF($htmlFile, $defaultDomain, $pdfFile);
		$pdf->setHtml2Ps('/usr/bin/html2ps');
		$pdf->setPs2Pdf('/usr/bin/ps2pdf');
		$pdf->setGetUrl('/usr/local/bin/curl -i');
		// Set headers/footers
		$pdf->setHeader('color', 'black');
		$pdf->setFooter('left', '');
		$pdf->setFooter('right', '$D');

		$pdf->setDefaultPath(BASE_PATH.'/kernel/admin_templates/');

		$result = $pdf->convert();

		// Check if the result was an error
		if (PEAR::isError($result)) {
		    $this->Application->ApplicationDie($result->getMessage());
		}
		else {
			$download_url = rtrim($this->Application->BaseURL(), '/') . EXPORT_BASE_PATH . '/tmp.pdf';
		    echo "PDF file created successfully: $result";
		    echo '<br />Click <a href="' . $download_url . '">here</a> to view the PDF file.';
		}
	}

	/**
	 * Occurs, when config was parsed, allows to change config data dynamically
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function OnAfterConfigRead(kEvent $event)
	{
		parent::OnAfterConfigRead($event);

		if (defined('IS_INSTALL') && IS_INSTALL) {
			return ;
		}

		$order_number = (int)$this->Application->ConfigValue('Comm_Order_Number_Format_P');
		$order_sub_number = (int)$this->Application->ConfigValue('Comm_Order_Number_Format_S');

		$calc_fields = $this->Application->getUnitOption($event->Prefix, 'CalculatedFields');
		foreach ($calc_fields as $special => $fields) {
			$calc_fields[$special]['OrderNumber'] = str_replace('6', $order_number, $calc_fields[$special]['OrderNumber']);
			$calc_fields[$special]['OrderNumber'] = str_replace('3', $order_sub_number, $calc_fields[$special]['OrderNumber']);
		}
		$this->Application->setUnitOption($event->Prefix, 'CalculatedFields', $calc_fields);

		$fields = $this->Application->getUnitOption($event->Prefix, 'Fields');
		$fields['Number']['format'] = str_replace('%06d', '%0'.$order_number.'d', $fields['Number']['format']);
		$fields['SubNumber']['format'] = str_replace('%03d', '%0'.$order_sub_number.'d', $fields['SubNumber']['format']);

		$site_helper = $this->Application->recallObject('SiteHelper');
		/* @var $site_helper SiteHelper */

		$fields['BillingCountry']['default'] = $site_helper->getDefaultCountry('Billing');
		$fields['ShippingCountry']['default'] = $site_helper->getDefaultCountry('Shipping');

		if (!$this->Application->isAdminUser) {
			$user_groups = explode(',', $this->Application->RecallVar('UserGroups'));
			$default_group = $this->Application->ConfigValue('User_LoggedInGroup');
			if (!in_array($default_group, $user_groups)){
				$user_groups[] = $default_group;
			}

			$sql_part = '';

			// limit payment types by domain
			$payment_types = $this->Application->siteDomainField('PaymentTypes');

			if (strlen($payment_types)) {
				$payment_types = explode('|', substr($payment_types, 1, -1));
				$sql_part .= ' AND PaymentTypeId IN (' . implode(',', $payment_types) . ')';
			}

			// limit payment types by user group
			$sql_part .= ' AND (PortalGroups LIKE "%%,'.implode(',%%" OR PortalGroups LIKE "%%,', $user_groups).',%%")';

			$fields['PaymentType']['options_sql'] = str_replace(
				'ORDER BY ',
				$sql_part . ' ORDER BY ',
				$fields['PaymentType']['options_sql']
			);
		}

		$this->Application->setUnitOption($event->Prefix, 'Fields', $fields);

		$user_forms = $this->Application->getUnitOption('u', 'Forms');

		$virtual_fields = $this->Application->getUnitOption($event->Prefix, 'VirtualFields');
		$virtual_fields['UserPassword']['hashing_method'] = $user_forms['default']['Fields']['PasswordHashingMethod']['default'];
		$this->Application->setUnitOption($event->Prefix, 'VirtualFields', $virtual_fields);
	}

	/**
	 * Allows configuring export options
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function OnBeforeExportBegin(kEvent $event)
	{
		parent::OnBeforeExportBegin($event);

		/** @var kDBItem $object */
		$object = $this->Application->recallObject($event->Prefix . '.export');

		$object->SetField('Number', 999999);
		$object->SetField('SubNumber', 999);
	}

	/**
	 * Returns specific to each item type columns only
	 *
	 * @param kEvent $event
	 * @return Array
	 * @access protected
	 */
	public function getCustomExportColumns(kEvent $event)
	{
		$columns = parent::getCustomExportColumns($event);

		$new_columns = Array (
			'__VIRTUAL__CustomerName' => 'CustomerName',
			'__VIRTUAL__TotalAmount' => 'TotalAmount',
			'__VIRTUAL__AmountWithoutVAT' =>	'AmountWithoutVAT',
			'__VIRTUAL__SubtotalWithDiscount' =>	'SubtotalWithDiscount',
			'__VIRTUAL__SubtotalWithoutDiscount' =>	'SubtotalWithoutDiscount',
			'__VIRTUAL__OrderNumber' => 'OrderNumber',
		);

		return array_merge($columns, $new_columns);
	}

	/**
	 * Saves content of temp table into live and
	 * redirects to event' default redirect (normally grid template)
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function OnSave(kEvent $event)
	{
		parent::OnSave($event);

		if ( $event->status != kEvent::erSUCCESS ) {
			return ;
		}

		foreach ( $this->trackCopiedOrderIDs($event) as $id ) {
			$this->Application->removeObject($event->getPrefixSpecial());

			$an_event = new kEvent($this->Prefix . ':Dummy');
			$this->Application->SetVar($this->Prefix . '_id', $id);
			$this->Application->SetVar($this->Prefix . '_mode', ''); // this is to fool ReserveItems to use live table
			$this->ReserveItems($an_event);
		}
	}

	/**
	 * Occurs after an item has been copied to live table
	 * Id of copied item is passed as event' 'id' param
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function OnAfterCopyToLive(kEvent $event)
	{
		parent::OnAfterCopyToLive($event);

		$this->trackCopiedOrderIDs($event, $event->getEventParam('id'));
	}

	/**
	 * Tracks copied order IDs.
	 *
	 * @param kEvent  $event Event.
	 * @param integer $id    Order ID.
	 *
	 * @return array
	 */
	protected function trackCopiedOrderIDs(kEvent $event, $id = null)
	{
		$setting_name = $event->Prefix . '_copied_ids' . $this->Application->GetVar('wid');
		$ids = $this->Application->GetVar($setting_name, array());

		if ( isset($id) ) {
			array_push($ids, $id);
			$this->Application->SetVar($setting_name, $ids);
		}

		return $ids;
	}

	/**
	 * Checks, that currently loaded item is allowed for viewing (non permission-based)
	 *
	 * @param kEvent $event
	 * @return bool
	 * @access protected
	 */
	protected function checkItemStatus(kEvent $event)
	{
		if ( $this->Application->isAdminUser ) {
			return true;
		}

		$object = $event->getObject();
		/* @var $object kDBItem */

		if ( !$object->isLoaded() ) {
			return true;
		}

		return $object->GetDBField('PortalUserId') == $this->Application->RecallVar('user_id');
	}

	// ===== Gift Certificates Related =====
	/**
	 * Enter description here...
	 *
	 * @param kEvent $event
	 */
	function OnApplyGiftCertificate($event)
	{
		$code = $this->Application->GetVar('giftcert_code');

		if ( $code == '' ) {
			return;
		}

		$object = $event->getObject();
		/* @var $object OrdersItem */

		$gift_certificate = $this->Application->recallObject('gc', null, Array ('skip_autoload' => true));
		/* @var $gift_certificate kDBItem */

		$gift_certificate->Load($code, 'Code');

		if ( !$gift_certificate->isLoaded() ) {
			$event->status = kEvent::erFAIL;
			$object->setCheckoutError(OrderCheckoutErrorType::GIFT_CERTIFICATE, OrderCheckoutError::GC_CODE_INVALID);
			$event->redirect = false; // check!!!

			return;
		}

		$debit = $gift_certificate->GetDBField('Debit');
		$expire_date = $gift_certificate->GetDBField('Expiration');

		if ( $gift_certificate->GetDBField('Status') != 1 || ($expire_date && $expire_date < adodb_mktime()) || ($debit <= 0) ) {
			$event->status = kEvent::erFAIL;
			$object->setCheckoutError(OrderCheckoutErrorType::GIFT_CERTIFICATE, OrderCheckoutError::GC_CODE_EXPIRED);
			$event->redirect = false;

			return;
		}

		$object->SetDBField('GiftCertificateId', $gift_certificate->GetDBField('GiftCertificateId'));
		$object->Update();

		$object->setCheckoutError(OrderCheckoutErrorType::GIFT_CERTIFICATE, OrderCheckoutError::GC_APPLIED);
	}

	/**
	 * Removes gift certificate from order
	 *
	 * @param kEvent $event
	 * @deprecated
	 */
	function OnRemoveGiftCertificate($event)
	{
		$object = $event->getObject();
		/* @var $object OrdersItem */

		$this->RemoveGiftCertificate($object);
		$object->setCheckoutError(OrderCheckoutErrorType::GIFT_CERTIFICATE, OrderCheckoutError::GC_REMOVED);

		$event->CallSubEvent('OnRecalculateItems');
	}

	function RemoveGiftCertificate(&$object)
	{
		$object->RemoveGiftCertificate();
	}

	function RecalculateGift($event)
	{
		$object = $event->getObject();
		/* @var $object OrdersItem */

		if ($object->GetDBField('Status') > ORDER_STATUS_PENDING) {
			return ;
		}
		$object->RecalculateGift($event);
	}

	function GetWholeOrderGiftCertificateDiscount($gift_certificate_id)
	{
		if (!$gift_certificate_id) {
			return 0;
		}

		$sql = 'SELECT Debit
				FROM '.TABLE_PREFIX.'GiftCertificates
				WHERE GiftCertificateId = '.$gift_certificate_id;
		return $this->Conn->GetOne($sql);
	}

	/**
	 * Downloads shipping tracking bar code, that was already generated by USPS service
	 *
	 * @param kEvent $event
	 */
	function OnDownloadLabel($event)
	{
		$event->status = kEvent::erSTOP;
		ini_set('memory_limit', '300M');
		ini_set('max_execution_time', '0');

		$object = $event->getObject();
		/* @var $object kDBItem */

		$file = $object->GetDBField('ShippingTracking') . '.pdf';
		$full_path = USPS_LABEL_FOLDER . $file;

		if ( !file_exists($full_path) || !is_file($full_path) ) {
			return;
		}

		$this->Application->setContentType(kUtil::mimeContentType($full_path), false);
		header('Content-Disposition: attachment; filename="' . $file . '"');
		readfile($full_path);
	}

	/**
	 * Occurs before validation attempt
	 *
	 * @param kEvent $event
	 * @return void
	 * @access protected
	 */
	protected function OnBeforeItemValidate(kEvent $event)
	{
		parent::OnBeforeItemValidate($event);

		$create_account = $this->Application->GetVar('create_account');

		$object = $event->getObject();
		/* @var $object kDBItem */

		$required_fields = Array ('UserPassword', 'UserPassword_plain', 'VerifyUserPassword', 'VerifyUserPassword_plain');
		$object->setRequired($required_fields, $create_account);

		$billing_email = $object->GetDBField('BillingEmail');

		if ( $create_account && $object->GetDBField('PortalUserId') == USER_GUEST && $billing_email ) {
			// check that e-mail available
			$sql = 'SELECT PortalUserId
					FROM ' . TABLE_PREFIX . 'Users
					WHERE Email = ' . $this->Conn->qstr($billing_email);
			$user_id = $this->Conn->GetOne($sql);

			if ( $user_id ) {
				$object->SetError('BillingEmail', 'unique');
			}
		}
	}

	/**
	 * Performs order update and returns results in format, needed by FormManager
	 *
	 * @param kEvent $event Event.
	 *
	 * @return void
	 */
	protected function OnUpdateAjax(kEvent $event)
	{
		/** @var AjaxFormHelper $ajax_form_helper */
		$ajax_form_helper = $this->Application->recallObject('AjaxFormHelper');
		$ajax_form_helper->transitEvent($event, 'OnUpdate');
	}

	/**
	 * Performs order update after billing step submission and returns results in format, needed by FormManager
	 *
	 * @param kEvent $event Event.
	 *
	 * @return void
	 */
	protected function OnProceedToPreviewAjax(kEvent $event)
	{
		/** @var AjaxFormHelper $ajax_form_helper */
		$ajax_form_helper = $this->Application->recallObject('AjaxFormHelper');
		$ajax_form_helper->transitEvent($event, 'OnProceedToPreview');
	}

}
